"""
API client for communicating with hosted FreeCAD MCP server.

Handles:
- Authentication and token management
- Command submission to hosted server
- WebSocket connection for real-time updates (Phase 3.2+)
- Automatic reconnection on failure
- Request/response serialization
"""

import asyncio
import json
import logging
from typing import Any, Dict, Optional
from urllib.parse import urljoin

import aiohttp

logger = logging.getLogger(__name__)


class HostedServerClient:
    """Client for communicating with hosted FreeCAD MCP server."""

    def __init__(self, server_url: str, api_key: str):
        """
        Initialize client.

        Args:
            server_url: Base URL of hosted server (e.g., "https://api.freecad-mcp.com")
            api_key: API key for authentication
        """
        self.server_url = server_url.rstrip("/")
        self.api_key = api_key
        self.session: Optional[aiohttp.ClientSession] = None
        self.is_authenticated = False
        self._auth_token = None

    async def authenticate(self) -> bool:
        """
        Authenticate with hosted server.

        In Phase 2, authentication is API key based.
        In Phase 3+, can exchange API key for session token.

        Returns:
            True if authentication successful
        """
        try:
            if not self.session:
                self.session = aiohttp.ClientSession()

            # Test authentication with /api/v1/auth/user endpoint
            headers = {"X-API-Key": self.api_key}
            url = urljoin(self.server_url, "/api/v1/auth/user")

            async with self.session.get(url, headers=headers, timeout=aiohttp.ClientTimeout(total=10)) as resp:
                if resp.status == 200:
                    user_data = await resp.json()
                    logger.info(f"Authenticated as user: {user_data.get('email')}")
                    self.is_authenticated = True
                    return True
                else:
                    logger.error(f"Authentication failed: {resp.status} {await resp.text()}")
                    return False

        except Exception as e:
            logger.error(f"Authentication error: {e}")
            return False

    async def execute_command(self, command: str, parameters: Dict = None) -> Dict:
        """
        Execute a CAD command on the hosted server.

        The server will:
        1. Check authentication & rate limits
        2. Gate premium operations
        3. Forward to local agent for execution
        4. Return result

        Args:
            command: Command name (e.g., "create_box", "move_object")
            parameters: Command parameters

        Returns:
            Server response dict with keys:
            - success: bool
            - result: dict or None
            - error: str or None
            - rate_limit_info: dict with usage info
        """
        if not self.is_authenticated:
            return {"success": False, "error": "Not authenticated. Call authenticate() first.", "result": None}

        try:
            if not self.session:
                self.session = aiohttp.ClientSession()

            headers = {"X-API-Key": self.api_key}
            url = urljoin(self.server_url, "/api/v1/execute")

            payload = {"command": command, "parameters": parameters or {}}

            async with self.session.post(
                url, json=payload, headers=headers, timeout=aiohttp.ClientTimeout(total=30)
            ) as resp:
                data = await resp.json()

                if resp.status == 200:
                    logger.debug(f"Command '{command}' executed successfully")
                    return data
                elif resp.status == 429:
                    logger.warning(f"Rate limit exceeded: {data.get('error')}")
                    return {"success": False, "error": f"Rate limit exceeded: {data.get('error')}", "result": None}
                elif resp.status == 403:
                    logger.warning(f"Access denied: {data.get('error')}")
                    return {"success": False, "error": f"Premium feature required: {data.get('error')}", "result": None}
                else:
                    logger.error(f"Command failed with status {resp.status}: {data}")
                    return {
                        "success": False,
                        "error": f"Server error: {data.get('error', 'Unknown error')}",
                        "result": None,
                    }

        except asyncio.TimeoutError:
            logger.error(f"Command '{command}' timed out")
            return {"success": False, "error": "Request timed out", "result": None}
        except Exception as e:
            logger.error(f"Command execution error: {e}")
            return {"success": False, "error": f"Client error: {str(e)}", "result": None}

    async def get_usage_info(self) -> Optional[Dict]:
        """
        Get user's usage information.

        Returns:
            Dict with:
            - user_id: str
            - tier: str
            - operations_used: int
            - operations_limit: int or None
            - reset_at: str (ISO format)
        """
        if not self.is_authenticated:
            logger.warning("Not authenticated")
            return None

        try:
            if not self.session:
                self.session = aiohttp.ClientSession()

            headers = {"X-API-Key": self.api_key}
            url = urljoin(self.server_url, "/api/v1/usage")

            async with self.session.get(url, headers=headers, timeout=aiohttp.ClientTimeout(total=10)) as resp:
                if resp.status == 200:
                    return await resp.json()
                else:
                    logger.error(f"Failed to get usage info: {resp.status}")
                    return None

        except Exception as e:
            logger.error(f"Error getting usage info: {e}")
            return None

    async def connect_websocket(self) -> bool:
        """
        Connect to WebSocket for real-time communication (Phase 3.2+).

        This enables:
        - Real-time command execution
        - Live streaming of responses
        - Bidirectional communication

        Returns:
            True if connection successful (not yet implemented)
        """
        logger.info("WebSocket support coming in Phase 3.2")
        return False

    async def close(self) -> None:
        """Clean up client resources."""
        if self.session:
            await self.session.close()
            self.session = None
        self.is_authenticated = False

    async def __aenter__(self):
        """Async context manager entry."""
        await self.authenticate()
        return self

    async def __aexit__(self, exc_type, exc_val, exc_tb):
        """Async context manager exit."""
        await self.close()


async def example_usage():
    """Example of using the HostedServerClient."""
    client = HostedServerClient(server_url="http://localhost:8000", api_key="test-api-key")

    try:
        # Authenticate
        if not await client.authenticate():
            print("Authentication failed")
            return

        # Get usage info
        usage = await client.get_usage_info()
        print(f"Usage: {usage}")

        # Execute a command
        result = await client.execute_command("ping", {})
        print(f"Ping result: {result}")

    finally:
        await client.close()


if __name__ == "__main__":
    # Requires aiohttp installed
    asyncio.run(example_usage())
