"""
Conjure FreeCAD Server - Thin Client

Socket server that runs inside FreeCAD and executes primitive CAD commands.
All orchestration and business logic happens on the hosted server.
This client only receives and executes low-level commands.

Supported command types:
- Primitives: create_box, create_cylinder, create_sphere, create_cone
- Booleans: boolean_fuse, boolean_cut, boolean_intersect
- Transforms: move_object, rotate_object, copy_object, delete_object
- Modifiers: create_fillet, create_chamfer
- Queries: get_state, get_object_details, list_objects, get_bounding_box
- Export: export_stl, export_step
- View: set_view, capture_view
- Script: run_script (escape hatch)
"""

import builtins
import contextlib
import json
import math
import queue
import socket
import sys
import threading
import time
from pathlib import Path

import FreeCAD as App

# Import logger for persistent file logging
try:
    from logger import (
        get_log_file_path,
        log_command,
        log_connection,
        log_crash,
        log_debug,
        log_error,
        log_exception,
        log_info,
        log_shutdown,
        log_startup,
        log_warning,
    )

    HAS_LOGGER = True
except ImportError:
    HAS_LOGGER = False

    # Fallback no-op functions
    def log_startup(*args, **kwargs):
        pass

    def log_shutdown(*args, **kwargs):
        pass

    def log_command(*args, **kwargs):
        pass

    def log_exception(*args, **kwargs):
        pass

    def log_crash(*args, **kwargs):
        pass

    def log_connection(*args, **kwargs):
        pass

    def log_info(*args, **kwargs):
        pass

    def log_warning(*args, **kwargs):
        pass

    def log_error(*args, **kwargs):
        pass

    def log_debug(*args, **kwargs):
        pass

    def get_log_file_path():
        return "N/A"


try:
    import FreeCADGui as Gui

    HAS_GUI = True
except ImportError:
    HAS_GUI = False

try:
    from PySide2 import QtCore

    HAS_QT = True
except ImportError:
    HAS_QT = False

# Add shared module to path for material client
_shared_path = Path(__file__).parent / "shared"
if str(_shared_path) not in sys.path:
    sys.path.insert(0, str(_shared_path))

try:
    from materials import MaterialsClient
except ImportError:
    MaterialsClient = None  # Will use fallback

try:
    from shared.standards import get_standards_library

    HAS_STANDARDS = True
except ImportError:
    HAS_STANDARDS = False

    def get_standards_library():
        return None


# Feedback client for orchestration learning
try:
    from shared.feedback import (
        FeedbackClient,
        get_feedback_client,
        init_feedback_client,
        stop_feedback_client,
    )

    HAS_FEEDBACK = True
except ImportError:
    HAS_FEEDBACK = False
    FeedbackClient = None

    def get_feedback_client():
        return None

    def init_feedback_client(*args, **kwargs):
        return None

    def stop_feedback_client():
        pass


# =============================================================================
# PLACEMENT OBSERVER - Notifies server when objects move
# =============================================================================


class PlacementObserver:
    """
    FreeCAD document observer that watches for Placement changes.

    When an object's placement changes, notifies the cloud server to propagate
    constraints through the dependency graph.
    """

    def __init__(self):
        self.enabled = False
        self._cloud_bridge = None
        self._last_positions: dict[str, tuple] = {}  # object_name -> (x, y, z)
        self._debounce_timer = None

    def set_cloud_bridge(self, bridge):
        """Set the cloud bridge for sending notifications."""
        self._cloud_bridge = bridge

    def slotChangedObject(self, obj, prop):
        """Called when any object property changes."""
        if not self.enabled or prop != "Placement":
            return

        if not hasattr(obj, "Placement"):
            return

        name = obj.Name
        new_pos = obj.Placement.Base
        new_tuple = (new_pos.x, new_pos.y, new_pos.z)

        old_tuple = self._last_positions.get(name)
        self._last_positions[name] = new_tuple

        # Skip if position didn't actually change (or first observation)
        if old_tuple is None or old_tuple == new_tuple:
            return

        # Notify server of placement change (debounced)
        self._notify_server(name, old_tuple, new_tuple)

    def _notify_server(self, object_name: str, old_pos: tuple, new_pos: tuple):
        """Send placement change notification to server."""
        if not self._cloud_bridge or not self._cloud_bridge.ws:
            return

        try:
            # Send propagate_changes command
            import json

            message = json.dumps(
                {
                    "type": "command",
                    "command": {
                        "type": "propagate_changes",
                        "params": {
                            "object_name": object_name,
                            "old_position": list(old_pos),
                            "new_position": list(new_pos),
                        },
                    },
                }
            )
            with self._cloud_bridge._ws_lock:
                self._cloud_bridge.ws.send(message)
            App.Console.PrintMessage(f"[Conjure] Propagating constraints for {object_name}\n")
        except Exception as e:
            App.Console.PrintWarning(f"[Conjure] Failed to propagate: {e}\n")

    def slotDeletedObject(self, obj):
        """Called when an object is deleted."""
        name = getattr(obj, "Name", None)
        if name and name in self._last_positions:
            del self._last_positions[name]


# Global placement observer instance
_placement_observer: PlacementObserver | None = None


def get_placement_observer() -> PlacementObserver:
    """Get or create the global placement observer."""
    global _placement_observer
    if _placement_observer is None:
        _placement_observer = PlacementObserver()
    return _placement_observer


def enable_constraint_propagation(enable: bool = True):
    """Enable or disable automatic constraint propagation on object moves."""
    observer = get_placement_observer()
    observer.enabled = enable

    if enable and App.ActiveDocument:
        # Register observer with active document
        App.ActiveDocument.addDocumentObserver(observer)
        App.Console.PrintMessage("[Conjure] Constraint propagation enabled\n")
    elif not enable and App.ActiveDocument:
        try:
            App.ActiveDocument.removeDocumentObserver(observer)
        except Exception:
            pass  # Observer may not be registered
        App.Console.PrintMessage("[Conjure] Constraint propagation disabled\n")


class ConjureServer:
    """Thin socket server for FreeCAD command execution."""

    def __init__(self, host="0.0.0.0", port=9876, server_url="http://localhost:8000"):
        self.host = host
        self.port = port
        self.server_url = server_url
        self.running = False
        self.socket = None
        self.client = None
        self.thread = None

        # Thread-safe operation queue
        self.operation_queue = queue.Queue()
        self.result_map = {}

        # Materials client for engineering materials
        self.materials_client = None
        if MaterialsClient:
            try:
                self.materials_client = MaterialsClient(server_url)
            except Exception as e:
                App.Console.PrintWarning(f"Conjure: Failed to initialize materials client: {e}\n")

        # Feedback client for orchestration learning
        self.feedback_client = None
        self.feedback_enabled = True  # Can be disabled via config

        # Start queue processor if Qt available
        self.process_timer = None
        if HAS_QT:
            try:
                self.process_timer = QtCore.QTimer()
                self.process_timer.timeout.connect(self._process_queue)
                self.process_timer.start(50)
            except Exception as e:
                App.Console.PrintWarning(f"Queue processor failed: {e}\n")

    def start(self):
        """Start the server in background thread."""
        self.running = True
        self.thread = threading.Thread(target=self._server_loop, daemon=True)
        self.thread.start()
        App.Console.PrintMessage(f"Conjure server started on {self.host}:{self.port}\n")
        log_startup(self.port, self.server_url)
        if HAS_LOGGER:
            App.Console.PrintMessage(f"Conjure logs: {get_log_file_path()}\n")

        # Start feedback client for orchestration learning
        if HAS_FEEDBACK and self.feedback_enabled:
            try:
                import uuid

                session_id = f"freecad_{uuid.uuid4().hex[:8]}"
                self.feedback_client = init_feedback_client(
                    server_url=self.server_url,
                    session_id=session_id,
                    enabled=True,
                )
                App.Console.PrintMessage("[Conjure] Feedback collection enabled\n")
            except Exception as e:
                App.Console.PrintWarning(f"[Conjure] Feedback client failed: {e}\n")
                self.feedback_client = None

    def stop(self):
        """Stop the server."""
        self.running = False
        if self.process_timer:
            self.process_timer.stop()
        if self.socket:
            with contextlib.suppress(builtins.BaseException):
                self.socket.close()
        if self.client:
            with contextlib.suppress(builtins.BaseException):
                self.client.close()

        # Stop feedback client and flush remaining data
        if self.feedback_client:
            try:
                stats = self.feedback_client.get_stats()
                stop_feedback_client()
                App.Console.PrintMessage(f"[Conjure] Feedback: sent={stats['sent']}, buffered={stats['buffered']}\n")
            except Exception:
                pass
            self.feedback_client = None

        App.Console.PrintMessage("Conjure server stopped\n")
        log_shutdown()

    def _server_loop(self):
        """Main server loop - accepts connections and handles commands."""
        self.socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        self.socket.bind((self.host, self.port))
        self.socket.listen(1)
        self.socket.settimeout(1.0)

        while self.running:
            try:
                self.client, addr = self.socket.accept()
                App.Console.PrintMessage(f"Client connected from {addr}\n")
                log_connection("ACCEPT", f"from {addr}")
                self._handle_client()
            except socket.timeout:
                continue
            except Exception as e:
                if self.running:
                    App.Console.PrintWarning(f"Server error: {e}\n")
                    log_exception("server_loop", e)

    # Read-only commands that don't modify FreeCAD documents - safe for direct execution
    READ_ONLY_COMMANDS = {
        "health_check",
        "get_state",
        "get_metrics",
        "get_usage",
        "get_config",
        "get_object_details",
        "list_objects",
        "get_bounding_box",
        "get_topology",
        "get_edges",
        "get_faces",
        "get_face_info",
        "find_face",
        "list_faces",
        "list_edges",
        "validate_geometry",
        "check_interference",
        "measure_distance",
        "measure_face_distance",
        "check_face_alignment",
        "get_relationships",
        "check_relationships",
        "detect_relationships",
        "suggest_relationships",
        "get_help",
        "list_standards",
        "search_standards",
        "get_standard",
        "get_gear",
        "list_gears",
        "get_gear_formulas",
        "get_material_design_guidelines",
        "calculate_thread_geometry",
        "calculate_polygon_geometry",
        "calculate_clearance_fit",
        "calculate_shrinkage_compensation",
        "get_arm_endpoint",
        "get_radial_positions",
        "get_socket_placement",
        "position_on_scalp",
        "get_all_electrode_positions",
        "reload_module",
    }

    def _handle_client(self):
        """Handle commands from connected client."""
        buffer = ""
        while self.running and self.client:
            try:
                data = self.client.recv(4096).decode("utf-8")
                if not data:
                    break

                buffer += data
                while "\n" in buffer:
                    line, buffer = buffer.split("\n", 1)
                    if line.strip():
                        # Parse command to check if it's read-only
                        try:
                            cmd = json.loads(line.strip())
                            cmd_type = cmd.get("type", "")
                            cmd_type = self.COMMAND_ALIASES.get(cmd_type, cmd_type)

                            if cmd_type in self.READ_ONLY_COMMANDS:
                                # Direct execution for read-only commands (faster)
                                response = self._execute_command(line.strip())
                            else:
                                # Queue write commands for main thread (thread-safe)
                                response = self._execute_command_queued(line.strip())
                        except json.JSONDecodeError:
                            response = self._execute_command_queued(line.strip())

                        self.client.send((json.dumps(response) + "\n").encode("utf-8"))
            except Exception as e:
                App.Console.PrintWarning(f"Client error: {e}\n")
                log_exception("handle_client", e)
                break

        if self.client:
            self.client.close()
            self.client = None
            log_connection("DISCONNECT", "client disconnected")

    # Command aliases for better discoverability
    COMMAND_ALIASES = {
        # Boolean operation aliases
        "boolean_union": "boolean_fuse",
        "union": "boolean_fuse",
        "fuse": "boolean_fuse",
        "boolean_subtract": "boolean_cut",
        "subtract": "boolean_cut",
        "cut": "boolean_cut",
        "boolean_difference": "boolean_cut",
        "difference": "boolean_cut",
        "intersect": "boolean_intersect",
        # Transform aliases
        "move": "move_object",
        "translate": "move_object",
        "rotate": "rotate_object",
        "copy": "copy_object",
        "delete": "delete_object",
        "remove": "delete_object",
        # Primitive aliases
        "box": "create_box",
        "cylinder": "create_cylinder",
        "sphere": "create_sphere",
        "cone": "create_cone",
        "torus": "create_torus",
        "wedge": "create_wedge",
        "prism": "create_prism",
        # Modifier aliases
        "fillet": "create_fillet",
        "chamfer": "create_chamfer",
        # High-level feature operation aliases
        "round_face": "fillet_face",
        "smooth_face": "fillet_face",
        "bevel_face": "chamfer_face",
        "hole": "create_hole",
        "drill": "create_hole",
        "pocket": "create_pocket",
        "recess": "create_pocket",
        # Smart placement aliases
        "align": "align_to_face",
        "snap_to": "align_to_face",
        "place_on": "place_on_face",
        "center": "center_on",
        # Analysis aliases
        "interference": "check_interference",
        "collides": "check_interference",
        "fit_check": "analyze_fit",
        "fits": "analyze_fit",
        # Query aliases
        "state": "get_state",
        "get_enhanced_state": "get_state",  # MCP bridge compatibility
        "bounding_box": "get_bounding_box",
        "bbox": "get_bounding_box",
        # Geometry reference aliases
        "topology": "get_topology",
        "edges": "get_edges",
        "faces": "get_faces",
        "face_info": "get_face_info",
        "find": "find_face",
        # Import/Export aliases
        "import_svg": "import_svg",
        # Document management aliases
        "clear": "clear_document",
        "reset": "clear_document",
        "new": "new_document",
        # Reference object aliases (list_references/get_reference_info are now SERVER-SIDE)
        "place_ref": "place_reference",
    }

    def _record_feedback(self, cmd_type: str, success: bool, duration_ms: float, error: str = None):
        """Record feedback for a command execution (non-blocking)."""
        if not self.feedback_client:
            return

        try:
            failure_category = None
            if not success and error:
                failure_category = self.feedback_client.categorize_failure(error)

            self.feedback_client.record(
                command_type=cmd_type,
                success=success,
                execution_time_ms=duration_ms,
                failure_category=failure_category,
            )
        except Exception:
            pass  # Don't let feedback recording break command execution

    def _execute_command(self, command_str):
        """Parse and execute a command."""
        start_time = time.time()
        cmd_type = "unknown"
        params = {}

        try:
            cmd = json.loads(command_str)
            cmd_type = cmd.get("type", "")
            params = cmd.get("params", {})

            # Resolve command aliases
            cmd_type = self.COMMAND_ALIASES.get(cmd_type, cmd_type)

            # Route to handler
            handler = getattr(self, f"_cmd_{cmd_type}", None)
            if handler:
                result = handler(params)
                duration_ms = (time.time() - start_time) * 1000
                success = result.get("status") == "success"
                log_command(cmd_type, params, duration_ms, success, result.get("error"))
                # Record feedback for orchestration learning
                self._record_feedback(cmd_type, success, duration_ms, result.get("error"))
                return result
            else:
                duration_ms = (time.time() - start_time) * 1000
                error = f"Unknown command: {cmd_type}"
                log_command(cmd_type, params, duration_ms, False, error)
                self._record_feedback(cmd_type, False, duration_ms, error)
                return {"status": "error", "error": error}
        except json.JSONDecodeError as e:
            duration_ms = (time.time() - start_time) * 1000
            error = f"Invalid JSON: {e}"
            log_command(cmd_type, params, duration_ms, False, error)
            self._record_feedback(cmd_type, False, duration_ms, error)
            return {"status": "error", "error": error}
        except Exception as e:
            duration_ms = (time.time() - start_time) * 1000
            log_exception(f"command {cmd_type}", e)
            log_crash(f"command {cmd_type}", e, self._get_doc_state_for_logging())
            self._record_feedback(cmd_type, False, duration_ms, str(e))
            return {"status": "error", "error": str(e)}

    def _execute_command_threadsafe(self, command_str, timeout=30.0):
        """
        Execute command on main GUI thread via Qt.

        FreeCAD operations must run on the main thread to avoid crashes.
        Uses QTimer.singleShot for reliable cross-thread invocation.

        Args:
            command_str: JSON command string
            timeout: Maximum time to wait for result (seconds)

        Returns:
            Command result dict
        """
        import threading
        import uuid as uuid_mod

        op_id = str(uuid_mod.uuid4())
        result_event = threading.Event()

        def run_on_main_thread():
            try:
                result = self._execute_command(command_str)
                self.result_map[op_id] = {"result": result, "done": True}
            except Exception as e:
                self.result_map[op_id] = {"result": {"status": "error", "error": str(e)}, "done": True}
            result_event.set()

        # Use QTimer.singleShot to invoke on main thread
        if HAS_QT:
            QtCore.QTimer.singleShot(0, run_on_main_thread)
        else:
            # Fallback to direct execution if Qt not available
            run_on_main_thread()

        # Wait for result
        if result_event.wait(timeout):
            result = self.result_map.pop(op_id, {}).get("result", {"status": "error", "error": "No result"})
            return result
        else:
            self.result_map.pop(op_id, None)
            return {"status": "error", "error": f"Command timed out after {timeout}s"}

    def _get_doc_state_for_logging(self):
        """Get minimal document state for crash logging."""
        try:
            doc = App.ActiveDocument
            if doc:
                return {
                    "document": doc.Name,
                    "object_count": len(doc.Objects),
                }
        except Exception:
            pass
        return None

    def _execute_command_queued(self, command_str, timeout=30.0):
        """
        Execute command on main thread via operation queue.

        This is the preferred thread-safe method. Commands are queued and
        processed by _process_queue which runs on a QTimer on the main thread.

        Args:
            command_str: JSON command string
            timeout: Maximum time to wait for result (seconds)

        Returns:
            Command result dict
        """
        import uuid as uuid_mod

        op_id = str(uuid_mod.uuid4())
        log_debug(f"_execute_command_queued: queuing op {op_id[:8]} - {command_str[:80]}")

        # Queue the operation for main thread execution
        self.operation_queue.put(
            {
                "id": op_id,
                "func": self._execute_command,
                "args": [command_str],
                "kwargs": {},
            }
        )

        log_debug(f"_execute_command_queued: waiting for op {op_id[:8]}")

        # Wait for result with polling
        start_time = time.time()
        while time.time() - start_time < timeout:
            if op_id in self.result_map and self.result_map[op_id].get("done"):
                result = self.result_map.pop(op_id)["result"]
                log_debug(f"_execute_command_queued: got result for op {op_id[:8]}")
                return result
            time.sleep(0.01)  # 10ms polling interval

        # Timeout - clean up and return error
        log_error(f"_execute_command_queued: TIMEOUT for op {op_id[:8]}")
        self.result_map.pop(op_id, None)
        return {"status": "error", "error": f"Command timed out after {timeout}s"}

    def _process_queue(self):
        """Process operations on main thread (called by QTimer)."""
        try:
            while not self.operation_queue.empty():
                op = self.operation_queue.get_nowait()
                op_id = op["id"]
                log_debug(f"_process_queue: processing op {op_id[:8]}")
                try:
                    result = op["func"](*op.get("args", []), **op.get("kwargs", {}))
                    self.result_map[op_id] = {"result": result, "done": True}
                    log_debug(f"_process_queue: completed op {op_id[:8]}")
                except Exception as e:
                    log_error(f"_process_queue: error in op {op_id[:8]}: {e}")
                    self.result_map[op_id] = {"result": {"status": "error", "error": str(e)}, "done": True}
        except Exception as e:
            log_error(f"_process_queue: outer error: {e}")

    def execute_threadsafe(self, command_type, params, timeout=30.0):
        """
        Execute a command in a thread-safe manner.

        Queues the operation for main thread execution and waits for result.
        This must be used when calling from non-GUI threads (cloud bridge, etc).

        Args:
            command_type: Command type string (e.g., "create_box")
            params: Command parameters dict
            timeout: Maximum time to wait for result (seconds)

        Returns:
            Command result dict
        """
        import time
        import uuid

        op_id = str(uuid.uuid4())

        # Create operation for queue
        def execute_cmd():
            return self._execute_command(json.dumps({"type": command_type, "params": params}))

        self.operation_queue.put(
            {
                "id": op_id,
                "func": execute_cmd,
                "args": [],
                "kwargs": {},
            }
        )

        # Wait for result with timeout
        start_time = time.time()
        while time.time() - start_time < timeout:
            if op_id in self.result_map and self.result_map[op_id].get("done"):
                result = self.result_map.pop(op_id)["result"]
                return result
            time.sleep(0.01)  # 10ms polling interval

        # Timeout - clean up and return error
        self.result_map.pop(op_id, None)
        return {"status": "error", "error": f"Command timed out after {timeout}s"}

    # ==================== Parameter Helpers ====================

    def _get_object_param(self, params, required=True):
        """
        Get object name from params with standardized naming.

        Supports: object_name (preferred), object, name (legacy)
        Returns tuple of (object_name, error_response or None)
        """
        # Preferred: object_name
        obj_name = params.get("object_name")

        # Fallback: object (used by some commands)
        if not obj_name:
            obj_name = params.get("object")

        # Legacy fallback: name (deprecated for input objects)
        if not obj_name:
            obj_name = params.get("name")

        if required and not obj_name:
            return None, {
                "status": "error",
                "error": "Missing required parameter: 'object_name'. Specify the object to operate on.",
            }

        return obj_name, None

    def _get_object(self, params, required=True):
        """
        Get FreeCAD object from params.

        Returns tuple of (object, error_response or None)
        """
        obj_name, error = self._get_object_param(params, required)
        if error:
            return None, error

        doc = self._get_doc()
        obj = doc.getObject(obj_name)

        if required and not obj:
            return None, {
                "status": "error",
                "error": f"Object '{obj_name}' not found in document.",
            }

        return obj, None

    def _require_params(self, params, *required_keys):
        """
        Check that required parameters are present.

        Returns error response dict if missing, None if all present.
        """
        missing = [k for k in required_keys if k not in params or params[k] is None]
        if missing:
            return {
                "status": "error",
                "error": f"Missing required parameter(s): {', '.join(missing)}",
            }
        return None

    # ==================== Status Commands ====================

    def _cmd_health_check(self, params):
        """Return health status."""
        return {"status": "success", "server": "conjure", "version": "0.1.0", "operations_count": len(self.result_map)}

    def _cmd_get_metrics(self, params):
        """Return metrics data."""
        return {"status": "success", "metrics": {}}

    def _cmd_get_config(self, params):
        """Return config data."""
        return {"status": "success", "config": {}}

    def _cmd_get_usage(self, params):
        """Return usage stats."""
        return {"status": "success", "tier": "Free", "used": 0, "limit": 5000}

    def _cmd_reload_module(self, params):
        """Hot-reload the conjure module without FreeCAD restart.

        This is useful during development to pick up code changes without
        restarting FreeCAD. Note that some changes (global state, class
        instances) may not fully reload.

        Usage: Send {"type": "reload_module"} to the socket.
        """
        import importlib
        import sys

        reloaded_modules = []
        errors = []

        # List of modules to reload (in dependency order)
        modules_to_reload = [
            "standards",
            "materials",
            "conjure",
        ]

        for mod_name in modules_to_reload:
            if mod_name in sys.modules:
                try:
                    importlib.reload(sys.modules[mod_name])
                    reloaded_modules.append(mod_name)
                except Exception as e:
                    errors.append(f"{mod_name}: {str(e)}")

        if errors:
            return {
                "status": "partial",
                "message": f"Reloaded {len(reloaded_modules)} modules with {len(errors)} errors",
                "reloaded": reloaded_modules,
                "errors": errors,
            }

        return {
            "status": "success",
            "message": f"Reloaded {len(reloaded_modules)} modules",
            "reloaded": reloaded_modules,
        }

    def _cmd_clear_document(self, params):
        """Clear all objects from document, optionally keeping specified ones.

        Parameters:
            keep: list of object names to keep (optional)
            confirm: must be True to execute (safety check)
        """
        if not params.get("confirm", False):
            return {
                "status": "error",
                "error": "Safety check: set confirm=true to clear document",
            }

        doc = App.ActiveDocument
        if not doc:
            return {"status": "success", "deleted": 0, "message": "No active document"}

        keep_names = set(params.get("keep", []))
        deleted = []

        # Get all object names first (avoid modifying while iterating)
        all_names = [obj.Name for obj in doc.Objects]

        for name in all_names:
            if name not in keep_names:
                try:
                    doc.removeObject(name)
                    deleted.append(name)
                except Exception:
                    pass  # Object may have dependencies

        doc.recompute()

        return {
            "status": "success",
            "deleted": len(deleted),
            "deleted_objects": deleted,
            "kept": list(keep_names) if keep_names else [],
        }

    def _cmd_new_document(self, params):
        """Create a new document, optionally closing the current one.

        Parameters:
            name: document name (default: "Untitled")
            close_current: close current document first (default: False)
        """
        name = params.get("name", "Untitled")
        close_current = params.get("close_current", False)

        if close_current and App.ActiveDocument:
            App.closeDocument(App.ActiveDocument.Name)

        doc = App.newDocument(name)
        return {"status": "success", "document": doc.Name}

    def _cmd_enable_propagation(self, params):
        """Enable or disable automatic constraint propagation.

        When enabled, moving an object automatically updates all constrained
        objects to maintain their relationships.

        Parameters:
            enable: True to enable, False to disable (default: True)
        """
        enable = params.get("enable", True)
        enable_constraint_propagation(enable)
        return {
            "status": "success",
            "propagation_enabled": enable,
            "message": f"Constraint propagation {'enabled' if enable else 'disabled'}",
        }

    # ==================== Reference Objects ====================
    # NOTE: Reference dimensions are stored on the SERVER (COMMON_ITEMS in ai/intent.py)
    # Client only handles FreeCAD-specific operations (creating geometry, setting visual props)
    # Workflow: AI calls server get_common_items → then calls client place_reference with dimensions

    def _cmd_place_reference(self, params):
        """Place a reference object in the workspace for designing around.

        This is a thin adapter - dimensions come from server's COMMON_ITEMS.
        AI should call server's get_common_items first to get dimensions.

        Parameters:
            name: object name (e.g., "REF_phone")
            width: width in mm (required)
            height: height in mm (required)
            thickness: thickness/depth in mm (required)
            position: [x, y, z] placement (default [0,0,0])
            angle: viewing angle in degrees (default 70)
            orientation: "portrait" or "landscape" (default portrait)
        """
        # Require dimensions - should come from server's COMMON_ITEMS
        width = params.get("width")
        height = params.get("height")
        thickness = params.get("thickness") or params.get("depth")

        if not all([width, height, thickness]):
            return {
                "status": "error",
                "error": "Missing dimensions. Use server's get_common_items to get device dimensions first.",
                "required": ["width", "height", "thickness"],
            }

        obj_name = params.get("name", "REF_object")
        orientation = params.get("orientation", "portrait")
        angle = params.get("angle", 70)  # Default viewing angle
        pos = params.get("position") or [0, 0, 0]

        # Swap dimensions for landscape
        if orientation == "landscape":
            width, height = height, width

        doc = self._get_doc()

        # Create the reference object as a box
        ref = doc.addObject("Part::Box", obj_name)
        ref.Length = width
        ref.Width = thickness
        ref.Height = height

        # Position and rotate to viewing angle
        ref.Placement.Base = App.Vector(pos[0], pos[1], pos[2])
        # Rotate around X axis for viewing angle (leaning back)
        # Negative angle so phone leans BACK (top away from viewer)
        ref.Placement.Rotation = App.Rotation(App.Vector(1, 0, 0), -(90 - angle))

        # Set visual properties to distinguish from design geometry
        if hasattr(ref, "ViewObject") and ref.ViewObject:
            ref.ViewObject.Transparency = 60
            ref.ViewObject.ShapeColor = (0.2, 0.6, 1.0)  # Light blue

        doc.recompute()

        return {
            "status": "success",
            "object": obj_name,
            "dimensions": {"width": width, "height": height, "thickness": thickness},
            "angle": angle,
            "orientation": orientation,
            "tip": f"Design your stand around this {width}x{height}x{thickness}mm reference",
        }

    # ==================== Primitive Commands ====================

    def _cmd_create_box(self, params):
        """Create a box primitive."""
        doc = self._get_doc()
        name = params.get("name", "Box")
        box = doc.addObject("Part::Box", name)
        box.Length = params.get("length", 10)
        box.Width = params.get("width", 10)
        box.Height = params.get("height", 10)
        pos = params.get("position") or [0, 0, 0]
        box.Placement.Base = App.Vector(pos[0], pos[1], pos[2])
        doc.recompute()
        return {"status": "success", "object": name}

    def _cmd_create_cylinder(self, params):
        """Create a cylinder."""
        doc = self._get_doc()
        name = params.get("name", "Cylinder")
        cyl = doc.addObject("Part::Cylinder", name)
        cyl.Radius = params.get("radius", 5)
        cyl.Height = params.get("height", 10)
        pos = params.get("position") or [0, 0, 0]
        cyl.Placement.Base = App.Vector(pos[0], pos[1], pos[2])
        doc.recompute()
        return {"status": "success", "object": name}

    def _cmd_create_sphere(self, params):
        """Create a sphere."""
        doc = self._get_doc()
        name = params.get("name", "Sphere")
        sphere = doc.addObject("Part::Sphere", name)
        sphere.Radius = params.get("radius", 5)
        pos = params.get("position") or [0, 0, 0]
        sphere.Placement.Base = App.Vector(pos[0], pos[1], pos[2])
        doc.recompute()
        return {"status": "success", "object": name}

    def _cmd_create_cone(self, params):
        """Create a cone."""
        doc = self._get_doc()
        name = params.get("name", "Cone")
        cone = doc.addObject("Part::Cone", name)
        cone.Radius1 = params.get("radius1", 5)
        cone.Radius2 = params.get("radius2", 0)
        cone.Height = params.get("height", 10)
        pos = params.get("position") or [0, 0, 0]
        cone.Placement.Base = App.Vector(pos[0], pos[1], pos[2])
        doc.recompute()
        return {"status": "success", "object": name}

    def _cmd_create_torus(self, params):
        """Create a torus."""
        doc = self._get_doc()
        name = params.get("name", "Torus")
        torus = doc.addObject("Part::Torus", name)
        torus.Radius1 = params.get("radius1", 10)
        torus.Radius2 = params.get("radius2", 2)
        pos = params.get("position") or [0, 0, 0]
        torus.Placement.Base = App.Vector(pos[0], pos[1], pos[2])
        doc.recompute()
        return {"status": "success", "object": name}

    def _cmd_create_wedge(self, params):
        """
        Create a wedge (angled block) - useful for sloped/angled surfaces.

        Parameters:
            name: Object name
            xmin, xmax: X range at base (default 0, 10)
            ymin, ymax: Y range (depth, default 0, 10)
            zmin, zmax: Z range (height, default 0, 10)
            x2min, x2max: X range at top (default same as base for rectangular wedge)
            position: [x, y, z] placement

        For a simple angled ramp, set x2min=x2max to create a triangular cross-section.
        """
        doc = self._get_doc()
        name = params.get("name", "Wedge")
        wedge = doc.addObject("Part::Wedge", name)

        # Base dimensions
        wedge.Xmin = params.get("xmin", 0)
        wedge.Xmax = params.get("xmax", 10)
        wedge.Ymin = params.get("ymin", 0)
        wedge.Ymax = params.get("ymax", 10)
        wedge.Zmin = params.get("zmin", 0)
        wedge.Zmax = params.get("zmax", 10)

        # Top dimensions (for angled/sloped top)
        wedge.X2min = params.get("x2min", params.get("xmin", 0))
        wedge.X2max = params.get("x2max", params.get("xmax", 10))
        wedge.Z2min = params.get("z2min", params.get("zmax", 10))
        wedge.Z2max = params.get("z2max", params.get("zmax", 10))

        pos = params.get("position") or [0, 0, 0]
        wedge.Placement.Base = App.Vector(pos[0], pos[1], pos[2])
        doc.recompute()
        return {"status": "success", "object": name}

    def _cmd_create_prism(self, params):
        """
        Create a prism with specified polygon base and height.

        Parameters:
            name: Object name
            polygon: Number of sides (default 6 for hexagon)
            circumradius: Radius of circumscribed circle (default 5)
            height: Height of prism (default 10)
            position: [x, y, z] placement
        """
        doc = self._get_doc()
        name = params.get("name", "Prism")
        prism = doc.addObject("Part::Prism", name)
        prism.Polygon = params.get("polygon", 6)
        prism.Circumradius = params.get("circumradius", 5)
        prism.Height = params.get("height", 10)
        pos = params.get("position") or [0, 0, 0]
        prism.Placement.Base = App.Vector(pos[0], pos[1], pos[2])
        doc.recompute()
        return {"status": "success", "object": name}

    # ==================== Gear Commands ====================
    # NOTE: All gear geometry is calculated on the SERVER (cad/gear_generator.py)
    # Client only receives pre-computed points and creates the shape

    def _cmd_create_gear_from_profile(self, params):
        """Create spur gear from server-computed profile points.

        Server calculates involute geometry - client creates smooth B-spline profile.
        Uses B-spline interpolation for smooth tooth curves instead of polygon facets.
        """
        import Part

        doc = self._get_doc()
        name = params.get("name", "Gear")
        points = params.get("points", [])
        height = params.get("height", 10)
        bore_d = params.get("bore_diameter", 0)
        pos = params.get("position") or [0, 0, 0]
        use_bspline = params.get("smooth", False)  # Disabled - use high-res polygon

        if not points:
            return {"status": "error", "error": "No profile points provided"}

        try:
            # Convert 2D points to FreeCAD vectors
            vectors = [App.Vector(p[0], p[1], 0) for p in points]

            # Create profile wire - high-resolution polygon with proper involute points
            if use_bspline and len(vectors) > 4:
                # B-spline interpolation (experimental - currently disabled)
                closed_vectors = vectors + [vectors[0]]
                bspline = Part.BSplineCurve()
                bspline.interpolate(closed_vectors)
                wire = Part.Wire([bspline.toShape()])
            else:
                # High-resolution polygon from server-computed involute points
                vectors.append(vectors[0])  # Close profile
                wire = Part.makePolygon(vectors)

            face = Part.Face(wire)
            gear_shape = face.extrude(App.Vector(0, 0, height))

            # Add center bore if specified
            if bore_d > 0:
                bore = Part.makeCylinder(bore_d / 2, height + 2, App.Vector(0, 0, -1))
                gear_shape = gear_shape.cut(bore)

            gear_obj = doc.addObject("Part::Feature", name)
            gear_obj.Shape = gear_shape
            gear_obj.Placement.Base = App.Vector(pos[0], pos[1], pos[2])
            doc.recompute()

            return {"status": "success", "object": name, "profile": "bspline" if use_bspline else "polygon"}
        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_create_internal_gear_from_profile(self, params):
        """Create internal (ring) gear from server-computed profile points.

        Server calculates involute tooth space profile - client creates smooth B-spline
        profile and cuts it from cylinder for smooth internal gear teeth.
        """
        import Part

        doc = self._get_doc()
        name = params.get("name", "RingGear")
        points = params.get("points", [])
        height = params.get("height", 10)
        outer_d = params.get("outer_diameter", 100)
        pos = params.get("position") or [0, 0, 0]
        use_bspline = params.get("smooth", False)  # Disabled - use high-res polygon

        if not points:
            return {"status": "error", "error": "No profile points provided"}

        try:
            # Create outer cylinder
            outer_cyl = Part.makeCylinder(outer_d / 2, height)

            # Convert 2D points to FreeCAD vectors
            vectors = [App.Vector(p[0], p[1], 0) for p in points]

            # Create tooth space profile - high-resolution polygon
            if use_bspline and len(vectors) > 4:
                closed_vectors = vectors + [vectors[0]]
                bspline = Part.BSplineCurve()
                bspline.interpolate(closed_vectors)
                wire = Part.Wire([bspline.toShape()])
            else:
                vectors.append(vectors[0])  # Close profile
                wire = Part.makePolygon(vectors)

            face = Part.Face(wire)
            tooth_cut = face.extrude(App.Vector(0, 0, height + 2))
            tooth_cut.translate(App.Vector(0, 0, -1))

            # Cut teeth from cylinder
            ring_gear = outer_cyl.cut(tooth_cut)

            gear_obj = doc.addObject("Part::Feature", name)
            gear_obj.Shape = ring_gear
            gear_obj.Placement.Base = App.Vector(pos[0], pos[1], pos[2])
            doc.recompute()

            return {"status": "success", "object": name, "profile": "bspline" if use_bspline else "polygon"}
        except Exception as e:
            return {"status": "error", "error": str(e)}

    # Legacy gear commands - delegate to profile-based commands
    # These exist for backward compatibility until MCP bridge is updated

    def _cmd_create_gear(self, params):
        """Legacy: Create gear (generates profile locally for now)."""
        # TODO: Remove once MCP bridge uses server-side generator
        import Part

        self._get_doc()
        name = params.get("name", "Gear")
        m = params.get("module", 1.0)
        z = params.get("teeth", 20)
        height = params.get("face_width", 10)
        bore_d = params.get("bore_diameter", 0)
        pos = params.get("position") or [0, 0, 0]

        # Generate profile locally (temporary until server handles this)
        pitch_r = m * z / 2
        tip_r = pitch_r + 0.7 * m
        root_r = pitch_r - 0.9 * m
        angular_pitch = 2 * math.pi / z
        tooth_angle = angular_pitch * 0.48

        points = []
        for tooth in range(z):
            theta = tooth * angular_pitch
            points.extend(
                [
                    (root_r * math.cos(theta - angular_pitch * 0.02), root_r * math.sin(theta - angular_pitch * 0.02)),
                    (root_r * math.cos(theta + angular_pitch * 0.02), root_r * math.sin(theta + angular_pitch * 0.02)),
                    (tip_r * math.cos(theta + tooth_angle * 0.35), tip_r * math.sin(theta + tooth_angle * 0.35)),
                    (tip_r * math.cos(theta + angular_pitch * 0.25), tip_r * math.sin(theta + angular_pitch * 0.25)),
                    (
                        tip_r * math.cos(theta + angular_pitch * 0.5 - tooth_angle * 0.35),
                        tip_r * math.sin(theta + angular_pitch * 0.5 - tooth_angle * 0.35),
                    ),
                    (root_r * math.cos(theta + angular_pitch * 0.48), root_r * math.sin(theta + angular_pitch * 0.48)),
                ]
            )

        return self._cmd_create_gear_from_profile(
            {"name": name, "points": points, "height": height, "bore_diameter": bore_d, "position": pos}
        )

    def _cmd_create_internal_gear(self, params):
        """Legacy: Create internal gear (generates profile locally for now)."""
        # TODO: Remove once MCP bridge uses server-side generator
        import Part

        self._get_doc()
        name = params.get("name", "RingGear")
        m = params.get("module", 1.0)
        z = params.get("teeth", 48)
        height = params.get("face_width", 10)
        wall = params.get("wall_thickness", 5)
        pos = params.get("position") or [0, 0, 0]

        pitch_r = m * z / 2
        inner_tip_r = pitch_r - 0.7 * m
        inner_root_r = pitch_r + 0.9 * m
        outer_r = inner_root_r + wall
        angular_pitch = 2 * math.pi / z
        tooth_angle = angular_pitch * 0.48

        points = []
        for tooth in range(z):
            theta = tooth * angular_pitch
            points.extend(
                [
                    (
                        inner_root_r * math.cos(theta - angular_pitch * 0.02),
                        inner_root_r * math.sin(theta - angular_pitch * 0.02),
                    ),
                    (
                        inner_root_r * math.cos(theta + angular_pitch * 0.02),
                        inner_root_r * math.sin(theta + angular_pitch * 0.02),
                    ),
                    (
                        inner_tip_r * math.cos(theta + tooth_angle * 0.35),
                        inner_tip_r * math.sin(theta + tooth_angle * 0.35),
                    ),
                    (
                        inner_tip_r * math.cos(theta + angular_pitch * 0.25),
                        inner_tip_r * math.sin(theta + angular_pitch * 0.25),
                    ),
                    (
                        inner_tip_r * math.cos(theta + angular_pitch * 0.5 - tooth_angle * 0.35),
                        inner_tip_r * math.sin(theta + angular_pitch * 0.5 - tooth_angle * 0.35),
                    ),
                    (
                        inner_root_r * math.cos(theta + angular_pitch * 0.48),
                        inner_root_r * math.sin(theta + angular_pitch * 0.48),
                    ),
                ]
            )

        return self._cmd_create_internal_gear_from_profile(
            {"name": name, "points": points, "height": height, "outer_diameter": outer_r * 2, "position": pos}
        )

    # ==================== Boolean Commands ====================

    def _cmd_boolean_fuse(self, params):
        """Fuse objects."""
        doc = self._get_doc()
        name = params.get("name", "Fused")
        # Support both formats: objects array or object_a/object_b
        objects = params.get("objects") or []
        if not objects:
            obj_a = params.get("object_a")
            obj_b = params.get("object_b")
            if obj_a and obj_b:
                objects = [obj_a, obj_b]

        shapes = []
        for obj_name in objects:
            obj = doc.getObject(obj_name)
            if obj and hasattr(obj, "Shape"):
                shapes.append(obj.Shape)

        if len(shapes) >= 2:
            result = shapes[0]
            for shape in shapes[1:]:
                result = result.fuse(shape)
            fused = doc.addObject("Part::Feature", name)
            fused.Shape = result
            doc.recompute()
            return {"status": "success", "object": name}
        return {"status": "error", "error": "Need at least 2 objects to fuse"}

    def _cmd_boolean_cut(self, params):
        """Cut one object from another."""
        doc = self._get_doc()
        name = params.get("name", "Cut")
        base = doc.getObject(params.get("base"))
        tool = doc.getObject(params.get("tool"))

        if base and tool:
            result = base.Shape.cut(tool.Shape)
            cut = doc.addObject("Part::Feature", name)
            cut.Shape = result
            if not params.get("keep_tool", False):
                doc.removeObject(tool.Name)
            doc.recompute()
            return {"status": "success", "object": name}
        return {"status": "error", "error": "Base or tool object not found"}

    def _cmd_boolean_intersect(self, params):
        """Intersect objects."""
        doc = self._get_doc()
        name = params.get("name", "Intersected")
        # Support both formats: objects array or object_a/object_b
        objects = params.get("objects") or []
        if not objects:
            obj_a = params.get("object_a")
            obj_b = params.get("object_b")
            if obj_a and obj_b:
                objects = [obj_a, obj_b]

        shapes = []
        for obj_name in objects:
            obj = doc.getObject(obj_name)
            if obj and hasattr(obj, "Shape"):
                shapes.append(obj.Shape)

        if len(shapes) >= 2:
            result = shapes[0]
            for shape in shapes[1:]:
                result = result.common(shape)
            intersected = doc.addObject("Part::Feature", name)
            intersected.Shape = result
            doc.recompute()
            return {"status": "success", "object": name}
        return {"status": "error", "error": "Need at least 2 objects to intersect"}

    # ==================== Profile-Based Operations ====================
    # These operations enable organic modeling beyond primitive shapes

    def _cmd_extrude(self, params):
        """
        Extrude a 2D profile into a 3D solid.

        Parameters:
            profile: Profile definition (type, parameters, or reference)
            distance: Extrusion distance in mm
            direction: "normal", "custom", or vector [x, y, z]
            taper_angle: Draft angle in degrees (default 0)
            is_cut: If True, creates a pocket/cut instead of boss
            name: Result object name

        Profile types:
            - rectangle: {width, height}
            - circle: {radius}
            - polygon: {sides, radius}
            - face: {object_name, face} - extrude existing face
        """
        doc = self._get_doc()
        name = params.get("name", "Extruded")
        profile = params.get("profile", {})
        distance = params.get("distance", 10)
        direction = params.get("direction", "normal")
        taper = params.get("taper_angle", 0)
        is_cut = params.get("is_cut", False)
        base_object = params.get("base_object")  # For cut operations

        try:
            # Create the profile wire/face
            profile_type = profile.get("type", "rectangle")
            profile_params = profile.get("parameters", {})
            position = profile.get("position", [0, 0, 0])
            normal = profile.get("normal", [0, 0, 1])

            # Build the profile shape
            if profile_type == "rectangle":
                w = profile_params.get("width", 20)
                h = profile_params.get("height", 20)
                # Create rectangle centered at origin
                pts = [
                    App.Vector(-w / 2, -h / 2, 0),
                    App.Vector(w / 2, -h / 2, 0),
                    App.Vector(w / 2, h / 2, 0),
                    App.Vector(-w / 2, h / 2, 0),
                ]
                wire = Part.makePolygon(pts + [pts[0]])
                face = Part.Face(wire)

            elif profile_type == "circle":
                r = profile_params.get("radius", 10)
                circle = Part.makeCircle(r)
                wire = Part.Wire(circle)
                face = Part.Face(wire)

            elif profile_type == "polygon":
                sides = profile_params.get("sides", 6)
                r = profile_params.get("radius", 10)
                import math

                pts = []
                for i in range(sides):
                    angle = 2 * math.pi * i / sides
                    pts.append(App.Vector(r * math.cos(angle), r * math.sin(angle), 0))
                wire = Part.makePolygon(pts + [pts[0]])
                face = Part.Face(wire)

            elif profile_type == "face":
                # Use existing face from object
                ref_obj = doc.getObject(profile.get("reference_object"))
                face_ref = profile.get("reference_face", "Face1")
                if ref_obj and hasattr(ref_obj, "Shape"):
                    face_idx = int(face_ref.replace("Face", "")) - 1
                    face = ref_obj.Shape.Faces[face_idx]
                else:
                    return {"status": "error", "error": "Cannot find face reference"}

            else:
                return {"status": "error", "error": f"Unknown profile type: {profile_type}"}

            # Position the profile
            pos_vec = App.Vector(*position)
            normal_vec = App.Vector(*normal).normalize()

            # Calculate rotation to align Z-up to normal
            z_axis = App.Vector(0, 0, 1)
            if not z_axis.isEqual(normal_vec, 1e-6):
                rot_axis = z_axis.cross(normal_vec)
                if rot_axis.Length > 1e-6:
                    import math

                    rot_angle = math.acos(z_axis.dot(normal_vec)) * 180 / math.pi
                    face.rotate(App.Vector(0, 0, 0), rot_axis, rot_angle)

            face.translate(pos_vec)

            # Determine extrusion direction
            if direction == "normal":
                ext_dir = normal_vec * distance
            elif isinstance(direction, list):
                ext_dir = App.Vector(*direction).normalize() * distance
            else:
                ext_dir = normal_vec * distance

            # Create the extrusion
            if taper != 0:
                # Tapered extrusion using Part.makeExtrudeBS or similar
                solid = face.extrude(ext_dir)  # Basic for now
            else:
                solid = face.extrude(ext_dir)

            # Handle cut vs boss
            if is_cut and base_object:
                base = doc.getObject(base_object)
                if base and hasattr(base, "Shape"):
                    result_shape = base.Shape.cut(solid)
                    result = doc.addObject("Part::Feature", name)
                    result.Shape = result_shape
                else:
                    return {"status": "error", "error": f"Base object '{base_object}' not found"}
            else:
                result = doc.addObject("Part::Feature", name)
                result.Shape = solid

            doc.recompute()

            return {
                "status": "success",
                "object": name,
                "operation": "extrude",
                "profile_type": profile_type,
                "distance": distance,
            }

        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_revolve(self, params):
        """
        Revolve a 2D profile around an axis to create a rotational solid.

        Parameters:
            profile: Profile definition (same as extrude)
            axis_point: [x, y, z] point on rotation axis
            axis_direction: [x, y, z] direction of rotation axis
            angle: Rotation angle in degrees (default 360 for full revolution)
            is_cut: If True, creates a cut instead of adding material
            name: Result object name
        """
        doc = self._get_doc()
        name = params.get("name", "Revolved")
        profile = params.get("profile", {})
        axis_point = params.get("axis_point", [0, 0, 0])
        axis_dir = params.get("axis_direction", [0, 0, 1])
        angle = params.get("angle", 360)
        is_cut = params.get("is_cut", False)
        base_object = params.get("base_object")

        try:
            # Create profile (similar to extrude)
            profile_type = profile.get("type", "rectangle")
            profile_params = profile.get("parameters", {})
            position = profile.get("position", [0, 0, 0])

            if profile_type == "rectangle":
                w = profile_params.get("width", 20)
                h = profile_params.get("height", 20)
                pts = [
                    App.Vector(-w / 2 + position[0], -h / 2 + position[1], position[2]),
                    App.Vector(w / 2 + position[0], -h / 2 + position[1], position[2]),
                    App.Vector(w / 2 + position[0], h / 2 + position[1], position[2]),
                    App.Vector(-w / 2 + position[0], h / 2 + position[1], position[2]),
                ]
                wire = Part.makePolygon(pts + [pts[0]])
                face = Part.Face(wire)

            elif profile_type == "circle":
                r = profile_params.get("radius", 10)
                circle = Part.makeCircle(r, App.Vector(*position))
                wire = Part.Wire(circle)
                face = Part.Face(wire)

            else:
                # For other types, create a simple rectangle as fallback
                w = profile_params.get("width", 20)
                h = profile_params.get("height", 20)
                pts = [
                    App.Vector(-w / 2 + position[0], -h / 2 + position[1], position[2]),
                    App.Vector(w / 2 + position[0], -h / 2 + position[1], position[2]),
                    App.Vector(w / 2 + position[0], h / 2 + position[1], position[2]),
                    App.Vector(-w / 2 + position[0], h / 2 + position[1], position[2]),
                ]
                wire = Part.makePolygon(pts + [pts[0]])
                face = Part.Face(wire)

            # Create revolution
            axis_pt = App.Vector(*axis_point)
            axis_vec = App.Vector(*axis_dir)

            solid = face.revolve(axis_pt, axis_vec, angle)

            if is_cut and base_object:
                base = doc.getObject(base_object)
                if base and hasattr(base, "Shape"):
                    result_shape = base.Shape.cut(solid)
                    result = doc.addObject("Part::Feature", name)
                    result.Shape = result_shape
                else:
                    return {"status": "error", "error": "Base object not found"}
            else:
                result = doc.addObject("Part::Feature", name)
                result.Shape = solid

            doc.recompute()

            return {
                "status": "success",
                "object": name,
                "operation": "revolve",
                "angle": angle,
            }

        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_loft(self, params):
        """
        Create a solid by blending between multiple profiles.

        Parameters:
            profiles: List of profile definitions with positions
            ruled: If True, use straight connections (faster)
            closed: If True, connect last profile to first
            solid: If True, create solid; False for surface
            name: Result object name

        Example profiles:
            [
                {"type": "circle", "parameters": {"radius": 20}, "position": [0, 0, 0]},
                {"type": "rectangle", "parameters": {"width": 30, "height": 30}, "position": [0, 0, 50]},
                {"type": "circle", "parameters": {"radius": 15}, "position": [0, 0, 100]}
            ]
        """
        doc = self._get_doc()
        name = params.get("name", "Lofted")
        profiles = params.get("profiles", [])
        ruled = params.get("ruled", False)
        closed = params.get("closed", False)
        solid = params.get("solid", True)

        if len(profiles) < 2:
            return {"status": "error", "error": "Loft requires at least 2 profiles"}

        try:
            wires = []

            for prof in profiles:
                profile_type = prof.get("type", "circle")
                profile_params = prof.get("parameters", {})
                position = prof.get("position", [0, 0, 0])
                normal = prof.get("normal", [0, 0, 1])

                if profile_type == "circle":
                    r = profile_params.get("radius", 10)
                    circle = Part.makeCircle(r, App.Vector(*position), App.Vector(*normal))
                    wire = Part.Wire(circle)

                elif profile_type == "rectangle":
                    w = profile_params.get("width", 20)
                    h = profile_params.get("height", 20)
                    # Create rectangle in XY plane, then position
                    pts = [
                        App.Vector(-w / 2, -h / 2, 0),
                        App.Vector(w / 2, -h / 2, 0),
                        App.Vector(w / 2, h / 2, 0),
                        App.Vector(-w / 2, h / 2, 0),
                    ]
                    wire = Part.makePolygon(pts + [pts[0]])
                    # Position and orient
                    wire.translate(App.Vector(*position))

                elif profile_type == "polygon":
                    sides = profile_params.get("sides", 6)
                    r = profile_params.get("radius", 10)
                    import math

                    pts = []
                    for i in range(sides):
                        angle = 2 * math.pi * i / sides
                        pts.append(
                            App.Vector(
                                r * math.cos(angle) + position[0], r * math.sin(angle) + position[1], position[2]
                            )
                        )
                    wire = Part.makePolygon(pts + [pts[0]])

                elif profile_type == "ellipse":
                    major = profile_params.get("major", 20)
                    minor = profile_params.get("minor", 10)
                    ellipse = Part.Ellipse(App.Vector(*position), major, minor)
                    wire = Part.Wire(Part.Edge(ellipse))

                else:
                    return {"status": "error", "error": f"Unknown profile type: {profile_type}"}

                wires.append(wire)

            # Create loft
            loft = Part.makeLoft(wires, solid, ruled, closed)

            result = doc.addObject("Part::Feature", name)
            result.Shape = loft
            doc.recompute()

            return {
                "status": "success",
                "object": name,
                "operation": "loft",
                "profile_count": len(profiles),
            }

        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_sweep(self, params):
        """
        Sweep a profile along a path to create a solid.

        Parameters:
            profile: Profile definition
            path: Path definition (type, parameters, or points)
            solid: If True, create solid; False for surface
            name: Result object name

        Path types:
            - line: {start: [x,y,z], end: [x,y,z]}
            - arc: {center, radius, start_angle, end_angle}
            - spline: {points: [[x,y,z], ...]}
            - helix: {radius, pitch, height}
            - edge: {object_name, edge} - use existing edge
        """
        doc = self._get_doc()
        name = params.get("name", "Swept")
        profile = params.get("profile", {})
        path = params.get("path", {})
        params.get("solid", True)

        try:
            import Part

            # Create profile wire
            profile_type = profile.get("type", "circle")
            profile_params = profile.get("parameters", {})
            profile_pos = profile.get("position", [0, 0, 0])

            if profile_type == "circle":
                r = profile_params.get("radius", 5)
                circle = Part.makeCircle(r, App.Vector(*profile_pos))
                profile_wire = Part.Wire(circle)

            elif profile_type == "rectangle":
                w = profile_params.get("width", 10)
                h = profile_params.get("height", 10)
                pts = [
                    App.Vector(-w / 2 + profile_pos[0], -h / 2 + profile_pos[1], profile_pos[2]),
                    App.Vector(w / 2 + profile_pos[0], -h / 2 + profile_pos[1], profile_pos[2]),
                    App.Vector(w / 2 + profile_pos[0], h / 2 + profile_pos[1], profile_pos[2]),
                    App.Vector(-w / 2 + profile_pos[0], h / 2 + profile_pos[1], profile_pos[2]),
                ]
                profile_wire = Part.makePolygon(pts + [pts[0]])

            else:
                r = profile_params.get("radius", 5)
                circle = Part.makeCircle(r, App.Vector(*profile_pos))
                profile_wire = Part.Wire(circle)

            # Create path
            path_type = path.get("type", "line")
            path_params = path.get("parameters", {})

            if path_type == "line":
                start = path_params.get("start", [0, 0, 0])
                end = path_params.get("end", [0, 0, 100])
                line = Part.makeLine(App.Vector(*start), App.Vector(*end))
                path_wire = Part.Wire(line)

            elif path_type == "arc":
                center = path_params.get("center", [0, 0, 0])
                radius = path_params.get("radius", 50)
                start_angle = path_params.get("start_angle", 0)
                end_angle = path_params.get("end_angle", 90)
                axis = path_params.get("axis", [0, 0, 1])
                import math

                arc = Part.makeCircle(radius, App.Vector(*center), App.Vector(*axis), start_angle, end_angle)
                path_wire = Part.Wire(arc)

            elif path_type == "spline":
                points = path.get("points", [[0, 0, 0], [50, 50, 50], [100, 0, 100]])
                pts = [App.Vector(*p) for p in points]
                spline = Part.BSplineCurve()
                spline.interpolate(pts)
                path_wire = Part.Wire(Part.Edge(spline))

            elif path_type == "helix":
                radius = path_params.get("radius", 20)
                pitch = path_params.get("pitch", 10)
                height = path_params.get("height", 50)
                center = path_params.get("center", [0, 0, 0])
                helix = Part.makeHelix(pitch, height, radius)
                helix.translate(App.Vector(*center))
                path_wire = Part.Wire(helix)

            elif path_type == "edge":
                ref_obj = doc.getObject(path.get("reference_object"))
                edge_ref = path.get("reference_edge", "Edge1")
                if ref_obj and hasattr(ref_obj, "Shape"):
                    edge_idx = int(edge_ref.replace("Edge", "")) - 1
                    path_wire = Part.Wire(ref_obj.Shape.Edges[edge_idx])
                else:
                    return {"status": "error", "error": "Edge reference not found"}

            else:
                return {"status": "error", "error": f"Unknown path type: {path_type}"}

            # Create sweep
            sweep = Part.Wire(profile_wire).makePipeShell([path_wire], True, False)

            if sweep.isNull():
                # Fallback to simpler sweep
                sweep = profile_wire.makePipe(path_wire)

            result = doc.addObject("Part::Feature", name)
            result.Shape = sweep
            doc.recompute()

            return {
                "status": "success",
                "object": name,
                "operation": "sweep",
                "path_type": path_type,
            }

        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_shell(self, params):
        """
        Hollow out a solid with uniform wall thickness.

        Parameters:
            object_name: Object to shell
            thickness: Wall thickness in mm
            faces_to_remove: List of faces to open (semantic or Face#)
            direction: "inward", "outward", or "both"
            name: Result object name

        Example:
            {"object_name": "Box", "thickness": 2, "faces_to_remove": ["top"]}
        """
        doc = self._get_doc()
        name = params.get("name", "Shelled")
        obj_name = params.get("object_name")
        thickness = params.get("thickness", 2)
        faces_to_remove = params.get("faces_to_remove", [])
        direction = params.get("direction", "inward")

        obj = doc.getObject(obj_name)
        if not obj or not hasattr(obj, "Shape"):
            return {"status": "error", "error": f"Object '{obj_name}' not found"}

        try:
            shape = obj.Shape

            # Resolve face references
            remove_faces = []
            for face_ref in faces_to_remove:
                if isinstance(face_ref, str):
                    if face_ref.startswith("Face"):
                        idx = int(face_ref.replace("Face", "")) - 1
                        if 0 <= idx < len(shape.Faces):
                            remove_faces.append(shape.Faces[idx])
                    else:
                        # Semantic face name
                        for _i, f in enumerate(shape.Faces):
                            normal = self._get_face_normal(f)
                            semantic = None
                            for fname, dir_vec in self.SEMANTIC_DIRECTIONS.items():
                                dot = sum(a * b for a, b in zip(normal, dir_vec))
                                if dot > 0.9:
                                    semantic = fname
                                    break
                            if semantic == face_ref.lower():
                                remove_faces.append(f)
                                break

            # Determine shell direction
            if direction == "outward":
                shell_thickness = thickness
            elif direction == "both":
                shell_thickness = thickness / 2
            else:  # inward
                shell_thickness = -thickness

            # Create shell
            if remove_faces:
                shelled = shape.makeThickness(remove_faces, shell_thickness, 1e-3)
            else:
                # Shell without removing faces (hollow sphere-like)
                shelled = shape.makeThickness([], shell_thickness, 1e-3)

            result = doc.addObject("Part::Feature", name)
            result.Shape = shelled
            doc.recompute()

            # Validate shell result
            if not result.Shape.isValid():
                doc.removeObject(name)
                return {
                    "status": "error",
                    "error": "Shell produced invalid geometry",
                    "suggestion": "Try reducing thickness or selecting different faces",
                }

            if result.Shape.Volume < 0.001:
                doc.removeObject(name)
                return {
                    "status": "error",
                    "error": "Shell produced near-zero volume",
                    "suggestion": "Geometry may be too thin for requested shell thickness",
                }

            return {
                "status": "success",
                "object": name,
                "operation": "shell",
                "thickness": thickness,
                "faces_removed": len(remove_faces),
                "volume": round(result.Shape.Volume, 2),
            }

        except Exception as e:
            return {
                "status": "error",
                "error": str(e),
                "suggestion": "Complex geometry may not support shell operation. Try simpler faces.",
            }

    def _cmd_import_svg(self, params):
        """
        Import an SVG file as a shape.

        Parameters:
            filepath: Path to SVG file
            name: Result object name
        """
        doc = self._get_doc()
        filepath = params.get("filepath")
        name = params.get("name", "SVGImport")

        if not filepath or not Path(filepath).exists():
            return {"status": "error", "error": f"File not found: {filepath}"}

        try:
            import importSVG

            existing_objects = set(doc.Objects)
            importSVG.insert(filepath, doc.Name)
            new_objects = [obj for obj in doc.Objects if obj not in existing_objects]

            if not new_objects:
                return {"status": "error", "error": "No objects imported from SVG"}

            # If name provided, rename the first (or combined) object
            if name:
                new_objects[0].Label = name

            return {
                "status": "success",
                "objects": [obj.Name for obj in new_objects],
                "primary_object": new_objects[0].Name,
                "count": len(new_objects),
            }
        except Exception as e:
            return {"status": "error", "error": str(e)}

    # ==================== Transform Commands ====================

    def _cmd_move_object(self, params):
        """Move an object."""
        obj, error = self._get_object(params)
        if error:
            return error

        doc = self._get_doc()
        x = params.get("x", 0)
        y = params.get("y", 0)
        z = params.get("z", 0)

        if params.get("relative", True):
            current = obj.Placement.Base
            obj.Placement.Base = App.Vector(current.x + x, current.y + y, current.z + z)
        else:
            obj.Placement.Base = App.Vector(x, y, z)

        doc.recompute()
        return {"status": "success", "object_name": obj.Name}

    def _cmd_rotate_object(self, params):
        """Rotate an object."""
        obj, error = self._get_object(params)
        if error:
            return error

        doc = self._get_doc()
        axis = params.get("axis", "z").lower()
        angle = params.get("angle", 0)
        center = params.get("center", [0, 0, 0])

        axis_map = {"x": App.Vector(1, 0, 0), "y": App.Vector(0, 1, 0), "z": App.Vector(0, 0, 1)}
        axis_vec = axis_map.get(axis, App.Vector(0, 0, 1))

        rotation = App.Rotation(axis_vec, angle)
        obj.Placement = App.Placement(
            obj.Placement.Base, rotation * obj.Placement.Rotation, App.Vector(center[0], center[1], center[2])
        )
        doc.recompute()
        return {"status": "success", "object_name": obj.Name}

    def _cmd_copy_object(self, params):
        """Copy an object."""
        # For copy, we use 'source' as the object to copy (different from standard object_name)
        doc = self._get_doc()
        source_name = params.get("source") or params.get("object_name")
        if not source_name:
            return {"status": "error", "error": "Missing required parameter: 'source' (object to copy)"}

        source = doc.getObject(source_name)
        if not source:
            return {"status": "error", "error": f"Source object '{source_name}' not found"}

        new_name = params.get("new_name", params.get("name", source.Name + "_copy"))
        offset = params.get("offset", [0, 0, 0])

        copy = doc.addObject("Part::Feature", new_name)
        copy.Shape = source.Shape.copy()
        copy.Placement.Base = App.Vector(
            source.Placement.Base.x + offset[0],
            source.Placement.Base.y + offset[1],
            source.Placement.Base.z + offset[2],
        )
        doc.recompute()
        return {"status": "success", "object_name": new_name}

    def _cmd_delete_object(self, params):
        """Delete an object."""
        obj_name, error = self._get_object_param(params)
        if error:
            return error

        doc = self._get_doc()
        if doc.getObject(obj_name):
            doc.removeObject(obj_name)
            doc.recompute()
            return {"status": "success", "deleted": obj_name}
        return {"status": "error", "error": f"Object '{obj_name}' not found"}

    # ==================== Modifier Commands ====================

    def _validate_shape(self, obj):
        """Check if an object has a valid shape after operation."""
        if not hasattr(obj, "Shape") or obj.Shape.isNull():
            return False, "Operation produced null shape"

        # Check for invalid bounding box (infinity values indicate failure)
        try:
            bbox = obj.Shape.BoundBox
            if bbox.XLength <= 0 or bbox.YLength <= 0 or bbox.ZLength <= 0:
                return False, "Operation produced empty or invalid geometry"
            # Check for infinity (indicates failed operation)
            if bbox.XMax > 1e300 or bbox.XMin < -1e300:
                return False, "Operation produced invalid geometry (computation failed)"
        except Exception as e:
            return False, f"Shape validation failed: {e}"

        return True, None

    def _cmd_create_fillet(self, params):
        """Create fillet on edges.

        Supports edge selection syntax:
          - List of indices: [1, 4, 7]
          - By face: {"face": "top"} or {"face": 3}
          - By type: {"type": "Circle"}
          - By length: {"length_gt": 10, "length_lt": 50}
          - Combined: {"face": "top", "type": "Line"}

        Options:
          - validate: (default True) Pre-validate edges, skip problematic ones
          - fallback: (default True) Try Part.makeFillet() if Part::Fillet fails
          - fallback_to_chamfer: (default True) Try chamfer if all fillet methods fail
        """
        obj, error = self._get_object(params)
        if error:
            return error

        doc = self._get_doc()
        radius = params.get("radius", 1)
        edges_param = params.get("edges", None)
        validate = params.get("validate", True)
        use_fallback = params.get("fallback", True)
        fallback_to_chamfer = params.get("fallback_to_chamfer", True)
        name = params.get("name", obj.Name + "_fillet")

        # Resolve edge selector (supports list, dict, or None for all)
        edges = self._resolve_edge_selector(obj, edges_param)

        if not edges:
            return {"status": "error", "error": "No edges match the specified criteria"}

        # Pre-validate edges if enabled
        skipped = []
        if validate:
            edges, skipped = self._validate_fillettable_edges(obj, edges, radius)
            if not edges:
                return {
                    "status": "error",
                    "error": "No valid edges after validation",
                    "skipped_edges": skipped,
                }

        # Try Part::Fillet feature first
        fillet = doc.addObject("Part::Fillet", name)
        fillet_name = fillet.Name  # Store name before potential deletion
        fillet.Base = obj
        fillet.Edges = [(i, radius, radius) for i in edges]
        doc.recompute()

        valid, error_msg = self._validate_shape(fillet)

        # If Part::Fillet fails and fallback is enabled, try Part.makeFillet()
        if not valid and use_fallback:
            doc.removeObject(fillet_name)

            try:
                # Part.makeFillet works on edge objects directly
                edge_objects = [obj.Shape.Edges[i - 1] for i in edges]
                new_shape = obj.Shape.makeFillet(radius, edge_objects)

                if not new_shape.isNull():
                    result = doc.addObject("Part::Feature", name)
                    result.Shape = new_shape
                    doc.recompute()

                    valid, error_msg = self._validate_shape(result)
                    if valid:
                        response = {
                            "status": "success",
                            "object_name": result.Name,
                            "method": "makeFillet",
                        }
                        if skipped:
                            response["skipped_edges"] = skipped
                        return response
                    else:
                        doc.removeObject(result.Name)
            except Exception as e:
                error_msg = f"Part::Fillet and makeFillet both failed: {e}"

        # If fillet failed and chamfer fallback is enabled, try chamfer
        if not valid and fallback_to_chamfer:
            try:
                chamfer = doc.addObject("Part::Chamfer", name)
                chamfer.Base = obj
                chamfer.Edges = [(i, radius, radius) for i in edges]
                doc.recompute()

                chamfer_valid, chamfer_error = self._validate_shape(chamfer)
                if chamfer_valid:
                    response = {
                        "status": "success",
                        "object_name": chamfer.Name,
                        "method": "chamfer_fallback",
                        "message": "Fillet failed, applied chamfer instead",
                    }
                    if skipped:
                        response["skipped_edges"] = skipped
                    return response
                else:
                    doc.removeObject(chamfer.Name)
            except Exception:
                pass  # Continue to error response

        if not valid:
            # Clean up fillet object if it still exists
            if fillet_name in [o.Name for o in doc.Objects]:
                doc.removeObject(fillet_name)
            return {
                "status": "error",
                "error": f"Fillet failed: {error_msg}. Try smaller radius or specify fewer edges.",
                "skipped_edges": skipped if skipped else None,
                "attempted_edges": edges,
                "chamfer_fallback_attempted": fallback_to_chamfer,
            }

        response = {"status": "success", "object_name": fillet_name, "method": "Part::Fillet"}
        if skipped:
            response["skipped_edges"] = skipped
        return response

    def _cmd_create_chamfer(self, params):
        """Create chamfer on edges.

        Supports edge selection syntax:
          - List of indices: [1, 4, 7]
          - By face: {"face": "top"} or {"face": 3}
          - By type: {"type": "Circle"}
          - By length: {"length_gt": 10, "length_lt": 50}
          - Combined: {"face": "top", "type": "Line"}

        Options:
          - validate: (default True) Pre-validate edges, skip problematic ones
        """
        obj, error = self._get_object(params)
        if error:
            return error

        doc = self._get_doc()
        size = params.get("size", 1)
        edges_param = params.get("edges", None)
        validate = params.get("validate", True)
        name = params.get("name", obj.Name + "_chamfer")

        # Resolve edge selector (supports list, dict, or None for all)
        edges = self._resolve_edge_selector(obj, edges_param)

        if not edges:
            return {"status": "error", "error": "No edges match the specified criteria"}

        # Pre-validate edges if enabled (use size as radius equivalent)
        skipped = []
        if validate:
            edges, skipped = self._validate_fillettable_edges(obj, edges, size)
            if not edges:
                return {
                    "status": "error",
                    "error": "No valid edges after validation",
                    "skipped_edges": skipped,
                }

        chamfer = doc.addObject("Part::Chamfer", name)
        chamfer.Base = obj
        chamfer.Edges = [(i, size, size) for i in edges]

        doc.recompute()

        # Validate result
        valid, error_msg = self._validate_shape(chamfer)
        if not valid:
            doc.removeObject(chamfer.Name)
            return {
                "status": "error",
                "error": f"Chamfer failed: {error_msg}. Try smaller size or specify fewer edges.",
                "skipped_edges": skipped if skipped else None,
                "attempted_edges": edges,
            }

        response = {"status": "success", "object_name": chamfer.Name}
        if skipped:
            response["skipped_edges"] = skipped
        return response

    # ==================== Query Commands ====================

    def _cmd_get_state(self, params):
        """Get document state."""
        doc = App.ActiveDocument
        if not doc:
            return {"status": "success", "document": None, "objects": []}

        objects = []
        for obj in doc.Objects:
            obj_info = {
                "name": obj.Name,
                "label": obj.Label,
                "type": obj.TypeId,
            }
            if hasattr(obj, "Shape"):
                bbox = obj.Shape.BoundBox
                obj_info["bounds"] = {
                    "min": [bbox.XMin, bbox.YMin, bbox.ZMin],
                    "max": [bbox.XMax, bbox.YMax, bbox.ZMax],
                }
            if hasattr(obj, "Placement"):
                pos = obj.Placement.Base
                obj_info["position"] = [pos.x, pos.y, pos.z]
            objects.append(obj_info)

        return {"status": "success", "document": doc.Name, "objects": objects}

    def _cmd_get_bounding_box(self, params):
        """Get bounding box of object."""
        obj, error = self._get_object(params)
        if error:
            return error

        if not hasattr(obj, "Shape"):
            return {"status": "error", "error": f"Object '{obj.Name}' has no shape"}

        bbox = obj.Shape.BoundBox
        return {
            "status": "success",
            "object_name": obj.Name,
            "bounding_box": {
                "min": [bbox.XMin, bbox.YMin, bbox.ZMin],
                "max": [bbox.XMax, bbox.YMax, bbox.ZMax],
                "center": [bbox.Center.x, bbox.Center.y, bbox.Center.z],
                "size": [bbox.XLength, bbox.YLength, bbox.ZLength],
            },
        }

    def _cmd_list_objects(self, params):
        """List all objects."""
        doc = App.ActiveDocument
        if not doc:
            return {"status": "success", "objects": []}

        pattern = params.get("pattern", "*")
        import fnmatch

        objects = []
        for obj in doc.Objects:
            if fnmatch.fnmatch(obj.Name, pattern):
                objects.append({"name": obj.Name, "label": obj.Label, "type": obj.TypeId})

        return {"status": "success", "objects": objects, "count": len(objects)}

    def _cmd_measure_distance(self, params):
        """Measure distance between two objects."""
        doc = self._get_doc()
        obj_a = doc.getObject(params.get("object_a"))
        obj_b = doc.getObject(params.get("object_b"))

        if not obj_a or not obj_b:
            return {"status": "error", "error": "One or both objects not found"}

        if hasattr(obj_a, "Shape") and hasattr(obj_b, "Shape"):
            dist = obj_a.Shape.distToShape(obj_b.Shape)[0]
            return {"status": "success", "distance": dist}
        return {"status": "error", "error": "Objects must have shapes"}

    # ==================== Geometry Reference System ====================

    # Semantic direction mapping for face lookup
    SEMANTIC_DIRECTIONS = {
        "top": [0, 0, 1],
        "bottom": [0, 0, -1],
        "front": [0, -1, 0],
        "back": [0, 1, 0],
        "left": [-1, 0, 0],
        "right": [1, 0, 0],
    }

    def _get_face_normal(self, face):
        """Extract normal vector from a face."""
        try:
            # Try to get normal at center of face
            uv = face.Surface.parameter(face.CenterOfMass)
            normal = face.normalAt(uv[0], uv[1])
            return [normal.x, normal.y, normal.z]
        except Exception:
            # Fallback for simple planar faces
            if hasattr(face.Surface, "Axis"):
                axis = face.Surface.Axis
                return [axis.x, axis.y, axis.z]
            return [0, 0, 0]

    def _get_face_edge_indices(self, shape, face):
        """Get 1-based edge indices for edges bounding a face."""
        face_edges = face.Edges
        indices = []
        for i, edge in enumerate(shape.Edges):
            for fe in face_edges:
                if edge.isSame(fe):
                    indices.append(i + 1)
                    break
        return indices

    def _get_face_by_direction(self, obj, direction, tolerance=0.1):
        """Find face index most aligned with semantic direction."""
        target = self.SEMANTIC_DIRECTIONS.get(direction.lower())
        if not target:
            return None

        best_face = None
        best_dot = -1

        for i, face in enumerate(obj.Shape.Faces):
            normal = self._get_face_normal(face)
            dot = sum(a * b for a, b in zip(normal, target))
            if dot > best_dot:
                best_dot = dot
                best_face = i + 1

        return best_face if best_dot > (1 - tolerance) else None

    def _get_edges_of_face(self, obj, face_ref):
        """Get edge indices for a face (by index or semantic name)."""
        shape = obj.Shape

        # Resolve semantic name to index
        if isinstance(face_ref, str):
            face_idx = self._get_face_by_direction(obj, face_ref)
            if not face_idx:
                return []
        else:
            face_idx = face_ref

        # Get face and its edge indices
        if face_idx < 1 or face_idx > len(shape.Faces):
            return []

        face = shape.Faces[face_idx - 1]
        return self._get_face_edge_indices(shape, face)

    def _get_edges_by_type(self, obj, edge_type):
        """Get edge indices matching a curve type."""
        indices = []
        for i, edge in enumerate(obj.Shape.Edges):
            if edge.Curve.__class__.__name__ == edge_type:
                indices.append(i + 1)
        return indices

    def _get_edges_by_length(self, obj, min_length=None, max_length=None):
        """Get edge indices within length range."""
        indices = []
        for i, edge in enumerate(obj.Shape.Edges):
            length = edge.Length
            if min_length is not None and length <= min_length:
                continue
            if max_length is not None and length >= max_length:
                continue
            indices.append(i + 1)
        return indices

    def _resolve_edge_selector(self, obj, selector):
        """Resolve edge selector to list of 1-based indices."""
        if selector is None:
            return list(range(1, len(obj.Shape.Edges) + 1))

        if isinstance(selector, list):
            # Convert Edge# strings to integers if needed
            result = []
            for item in selector:
                if isinstance(item, int):
                    result.append(item)
                elif isinstance(item, str) and item.startswith("Edge"):
                    try:
                        idx = int(item.replace("Edge", ""))
                        result.append(idx)
                    except ValueError:
                        pass  # Skip invalid edge refs
                elif isinstance(item, str):
                    # Try parsing as integer string
                    try:
                        result.append(int(item))
                    except ValueError:
                        pass
            return result if result else list(range(1, len(obj.Shape.Edges) + 1))

        if isinstance(selector, dict):
            result = set(range(1, len(obj.Shape.Edges) + 1))

            if "face" in selector:
                face_edges = self._get_edges_of_face(obj, selector["face"])
                result &= set(face_edges)

            if "type" in selector:
                type_edges = self._get_edges_by_type(obj, selector["type"])
                result &= set(type_edges)

            if "length_gt" in selector or "length_lt" in selector:
                length_edges = self._get_edges_by_length(
                    obj,
                    min_length=selector.get("length_gt"),
                    max_length=selector.get("length_lt"),
                )
                result &= set(length_edges)

            return sorted(result)

        return []

    def _validate_fillettable_edges(self, obj, edges, radius):
        """Pre-validate edges for fillet operation.

        Returns (valid_edges, skipped_edges) where skipped_edges contains
        details about why each edge was skipped.
        """
        valid = []
        skipped = []

        for idx in edges:
            if idx < 1 or idx > len(obj.Shape.Edges):
                skipped.append({"index": idx, "reason": "invalid_index"})
                continue

            edge = obj.Shape.Edges[idx - 1]

            # Check 1: Edge length must be > 2*radius to avoid self-intersection
            if edge.Length < 2 * radius:
                skipped.append(
                    {
                        "index": idx,
                        "reason": "too_short",
                        "length": round(edge.Length, 2),
                        "min_required": round(2 * radius, 2),
                    }
                )
                continue

            # Check 2: Skip BSpline edges (often problematic with fillets)
            curve_type = edge.Curve.__class__.__name__
            if curve_type == "BSplineCurve":
                skipped.append({"index": idx, "reason": "bspline_curve"})
                continue

            valid.append(idx)

        return valid, skipped

    def _cmd_get_topology(self, params):
        """Get complete topology breakdown of an object."""
        obj, error = self._get_object(params)
        if error:
            return error

        if not hasattr(obj, "Shape"):
            return {"status": "error", "error": f"Object '{obj.Name}' has no shape"}

        shape = obj.Shape

        vertices = []
        for i, v in enumerate(shape.Vertexes):
            vertices.append(
                {
                    "index": i + 1,
                    "position": [v.X, v.Y, v.Z],
                }
            )

        edges = []
        for i, e in enumerate(shape.Edges):
            edge_info = {
                "index": i + 1,
                "type": e.Curve.__class__.__name__,
                "length": round(e.Length, 4),
                "center": [
                    round(e.BoundBox.Center.x, 4),
                    round(e.BoundBox.Center.y, 4),
                    round(e.BoundBox.Center.z, 4),
                ],
            }
            if len(e.Vertexes) >= 2:
                edge_info["start"] = [e.Vertexes[0].X, e.Vertexes[0].Y, e.Vertexes[0].Z]
                edge_info["end"] = [e.Vertexes[-1].X, e.Vertexes[-1].Y, e.Vertexes[-1].Z]
            if hasattr(e.Curve, "Radius"):
                edge_info["radius"] = round(e.Curve.Radius, 4)
            edges.append(edge_info)

        faces = []
        for i, f in enumerate(shape.Faces):
            normal = self._get_face_normal(f)
            # Determine semantic direction
            semantic = None
            for name, direction in self.SEMANTIC_DIRECTIONS.items():
                dot = sum(a * b for a, b in zip(normal, direction))
                if dot > 0.9:
                    semantic = name
                    break

            faces.append(
                {
                    "index": i + 1,
                    "type": f.Surface.__class__.__name__,
                    "area": round(f.Area, 4),
                    "center": [
                        round(f.CenterOfMass.x, 4),
                        round(f.CenterOfMass.y, 4),
                        round(f.CenterOfMass.z, 4),
                    ],
                    "normal": [round(n, 4) for n in normal],
                    "semantic": semantic,
                    "edge_indices": self._get_face_edge_indices(shape, f),
                }
            )

        return {
            "status": "success",
            "object_name": obj.Name,
            "topology": {
                "vertex_count": len(vertices),
                "edge_count": len(edges),
                "face_count": len(faces),
                "vertices": vertices,
                "edges": edges,
                "faces": faces,
            },
        }

    def _cmd_get_edges(self, params):
        """Query edges by criteria."""
        obj, error = self._get_object(params)
        if error:
            return error

        if not hasattr(obj, "Shape"):
            return {"status": "error", "error": f"Object '{obj.Name}' has no shape"}

        # Build selector from params
        selector = {}
        if "face" in params:
            selector["face"] = params["face"]
        if "type" in params:
            selector["type"] = params["type"]
        if "length_gt" in params:
            selector["length_gt"] = params["length_gt"]
        if "length_lt" in params:
            selector["length_lt"] = params["length_lt"]

        if not selector:
            indices = list(range(1, len(obj.Shape.Edges) + 1))
        else:
            indices = self._resolve_edge_selector(obj, selector)

        # Return edge details
        edges = []
        for idx in indices:
            e = obj.Shape.Edges[idx - 1]
            edge_info = {
                "index": idx,
                "type": e.Curve.__class__.__name__,
                "length": round(e.Length, 4),
            }
            if hasattr(e.Curve, "Radius"):
                edge_info["radius"] = round(e.Curve.Radius, 4)
            edges.append(edge_info)

        return {
            "status": "success",
            "object_name": obj.Name,
            "edges": edges,
            "count": len(edges),
        }

    def _cmd_get_faces(self, params):
        """Query faces by criteria."""
        obj, error = self._get_object(params)
        if error:
            return error

        if not hasattr(obj, "Shape"):
            return {"status": "error", "error": f"Object '{obj.Name}' has no shape"}

        shape = obj.Shape

        # Filters
        face_type = params.get("type")
        direction = params.get("direction")  # Semantic: "top", "bottom", etc.
        min_area = params.get("area_gt")
        max_area = params.get("area_lt")

        faces = []
        for i, f in enumerate(shape.Faces):
            # Type filter
            if face_type and f.Surface.__class__.__name__ != face_type:
                continue

            # Area filter
            if min_area is not None and f.Area <= min_area:
                continue
            if max_area is not None and f.Area >= max_area:
                continue

            normal = self._get_face_normal(f)

            # Direction filter
            if direction:
                target = self.SEMANTIC_DIRECTIONS.get(direction.lower())
                if target:
                    dot = sum(a * b for a, b in zip(normal, target))
                    if dot < 0.9:
                        continue

            # Determine semantic direction
            semantic = None
            for name, dir_vec in self.SEMANTIC_DIRECTIONS.items():
                dot = sum(a * b for a, b in zip(normal, dir_vec))
                if dot > 0.9:
                    semantic = name
                    break

            faces.append(
                {
                    "index": i + 1,
                    "type": f.Surface.__class__.__name__,
                    "area": round(f.Area, 4),
                    "center": [
                        round(f.CenterOfMass.x, 4),
                        round(f.CenterOfMass.y, 4),
                        round(f.CenterOfMass.z, 4),
                    ],
                    "normal": [round(n, 4) for n in normal],
                    "semantic": semantic,
                    "edge_count": len(f.Edges),
                }
            )

        return {
            "status": "success",
            "object_name": obj.Name,
            "faces": faces,
            "count": len(faces),
        }

    # Aliases for MCP tool names (MCP uses list_*, socket server uses get_*)
    def _cmd_list_faces(self, params):
        """Alias for get_faces - MCP compatibility."""
        return self._cmd_get_faces(params)

    def _cmd_list_edges(self, params):
        """Alias for get_edges - MCP compatibility."""
        return self._cmd_get_edges(params)

    def _cmd_get_object_details(self, params):
        """Get detailed object information."""
        return self._cmd_get_topology(params)

    # ==================== High-Level Feature Operations ====================
    # These enable context-aware modeling: "fillet the top face" instead of "fillet edges 1,3,5,7"

    def _cmd_fillet_face(self, params):
        """
        Apply fillet to all edges of a semantic face.

        This is the key "quality modeling" command - instead of guessing edge indices,
        you simply specify "top", "front", etc. and the fillet is applied intelligently.

        Parameters:
            object_name: Target object
            face: Semantic face ("top", "bottom", "front", "back", "left", "right") or Face index
            radius: Fillet radius
            edge_filter: Optional - "Line" for straight edges only, "Circle" for arcs
            name: Optional result name

        Examples:
            {"command": "fillet_face", "parameters": {"object_name": "Box", "face": "top", "radius": 2}}
            {"command": "fillet_face", "parameters": {"object_name": "Box", "face": "top", "radius": 2, "edge_filter": "Line"}}
        """
        obj, error = self._get_object(params)
        if error:
            return error

        face = params.get("face")
        if not face:
            return {"status": "error", "error": "Missing required parameter: 'face'"}

        radius = params.get("radius", 1)
        edge_filter = params.get("edge_filter")
        name = params.get("name")

        # Get edges of the specified face
        edge_indices = self._get_edges_of_face(obj, face)
        if not edge_indices:
            return {"status": "error", "error": f"No edges found for face '{face}'"}

        # Apply edge type filter if specified
        if edge_filter:
            type_edges = self._get_edges_by_type(obj, edge_filter)
            edge_indices = [i for i in edge_indices if i in type_edges]
            if not edge_indices:
                return {"status": "error", "error": f"No '{edge_filter}' edges on face '{face}'"}

        # Delegate to fillet with the resolved edges
        fillet_params = {
            "object_name": obj.Name,
            "edges": edge_indices,
            "radius": radius,
        }
        if name:
            fillet_params["name"] = name

        result = self._cmd_create_fillet(fillet_params)

        # Enhance result with semantic info
        if result.get("status") == "success":
            result["face"] = face
            result["edge_count"] = len(edge_indices)
        return result

    def _cmd_chamfer_face(self, params):
        """
        Apply chamfer to all edges of a semantic face.

        Parameters:
            object_name: Target object
            face: Semantic face ("top", "bottom", etc.) or Face index
            size: Chamfer size
            edge_filter: Optional - "Line" for straight edges only
            name: Optional result name
        """
        obj, error = self._get_object(params)
        if error:
            return error

        face = params.get("face")
        if not face:
            return {"status": "error", "error": "Missing required parameter: 'face'"}

        size = params.get("size", 1)
        edge_filter = params.get("edge_filter")
        name = params.get("name")

        edge_indices = self._get_edges_of_face(obj, face)
        if not edge_indices:
            return {"status": "error", "error": f"No edges found for face '{face}'"}

        if edge_filter:
            type_edges = self._get_edges_by_type(obj, edge_filter)
            edge_indices = [i for i in edge_indices if i in type_edges]
            if not edge_indices:
                return {"status": "error", "error": f"No '{edge_filter}' edges on face '{face}'"}

        chamfer_params = {
            "object_name": obj.Name,
            "edges": edge_indices,
            "size": size,
        }
        if name:
            chamfer_params["name"] = name

        result = self._cmd_create_chamfer(chamfer_params)
        if result.get("status") == "success":
            result["face"] = face
            result["edge_count"] = len(edge_indices)
        return result

    def _cmd_create_hole(self, params):
        """
        Create a circular hole on a specific face.

        This is context-aware hole creation - specify "top" instead of calculating
        exact positions.

        Parameters:
            object_name: Target object to drill into
            face: Face to drill on ("top", "front", or Face index)
            diameter: Hole diameter
            depth: Hole depth
            offset: Optional [x, y] offset from face center
            name: Optional result name

        Example:
            {"command": "create_hole", "parameters": {
                "object_name": "Block",
                "face": "top",
                "diameter": 10,
                "depth": 15
            }}
        """
        obj, error = self._get_object(params)
        if error:
            return error

        face = params.get("face")
        if not face:
            return {"status": "error", "error": "Missing required parameter: 'face'"}

        diameter = params.get("diameter", 5)
        depth = params.get("depth", 10)
        offset = params.get("offset", [0, 0])
        result_name = params.get("name", f"{obj.Name}_WithHole")

        # Get face info
        face_idx = self._get_face_by_direction(obj, face) if isinstance(face, str) else face
        if not face_idx:
            return {"status": "error", "error": f"Could not find face '{face}'"}

        shape = obj.Shape
        if face_idx < 1 or face_idx > len(shape.Faces):
            return {"status": "error", "error": f"Face index {face_idx} out of range"}

        face_obj = shape.Faces[face_idx - 1]
        center = face_obj.CenterOfMass
        normal = self._get_face_normal(face_obj)

        try:
            doc = self._get_doc()

            # Create cylinder for the hole
            cyl_name = f"{obj.Name}_HoleTool"
            cyl = doc.addObject("Part::Cylinder", cyl_name)
            cyl.Radius = diameter / 2
            cyl.Height = depth + 0.1  # Small extra to ensure clean cut

            # Position: center of face, with optional offset
            # Note: offset is applied in local face coordinates (simplified for now)
            hole_pos = App.Vector(center.x + offset[0], center.y + offset[1], center.z)

            # Align cylinder: default Z-up, we need to point INTO the face
            normal_vec = App.Vector(normal[0], normal[1], normal[2])
            drill_dir = normal_vec.negative()

            z_axis = App.Vector(0, 0, 1)
            rotation = App.Rotation(z_axis, drill_dir)

            # Position slightly above face to ensure clean cut
            start_pos = hole_pos + normal_vec * 0.05
            cyl.Placement = App.Placement(start_pos, rotation)

            doc.recompute()

            # Boolean cut
            result = obj.Shape.cut(cyl.Shape)
            cut_obj = doc.addObject("Part::Feature", result_name)
            cut_obj.Shape = result

            # Clean up
            doc.removeObject(cyl_name)
            doc.recompute()

            return {
                "status": "success",
                "object_name": result_name,
                "face": face,
                "diameter": diameter,
                "depth": depth,
            }

        except Exception as e:
            return {"status": "error", "error": f"Failed to create hole: {e}"}

    def _cmd_create_pocket(self, params):
        """
        Create a rectangular pocket on a specific face.

        Parameters:
            object_name: Target object
            face: Face to pocket ("top", "front", or Face index)
            width: Pocket width
            length: Pocket length
            depth: Pocket depth
            offset: Optional [x, y] offset from face center
            name: Optional result name

        Example:
            {"command": "create_pocket", "parameters": {
                "object_name": "Block",
                "face": "top",
                "width": 20,
                "length": 30,
                "depth": 5
            }}
        """
        obj, error = self._get_object(params)
        if error:
            return error

        face = params.get("face")
        if not face:
            return {"status": "error", "error": "Missing required parameter: 'face'"}

        width = params.get("width", 10)
        length = params.get("length", 10)
        depth = params.get("depth", 5)
        offset = params.get("offset", [0, 0])
        result_name = params.get("name", f"{obj.Name}_WithPocket")

        face_idx = self._get_face_by_direction(obj, face) if isinstance(face, str) else face
        if not face_idx:
            return {"status": "error", "error": f"Could not find face '{face}'"}

        shape = obj.Shape
        if face_idx < 1 or face_idx > len(shape.Faces):
            return {"status": "error", "error": f"Face index {face_idx} out of range"}

        face_obj = shape.Faces[face_idx - 1]
        center = face_obj.CenterOfMass
        normal = self._get_face_normal(face_obj)

        try:
            doc = self._get_doc()

            # Create box for the pocket
            box_name = f"{obj.Name}_PocketTool"
            box = doc.addObject("Part::Box", box_name)
            box.Length = length
            box.Width = width
            box.Height = depth + 0.1

            # Align box to face normal
            normal_vec = App.Vector(normal[0], normal[1], normal[2])
            drill_dir = normal_vec.negative()
            z_axis = App.Vector(0, 0, 1)
            rotation = App.Rotation(z_axis, drill_dir)

            # Position box centered on face
            # Offset by half dimensions to center
            local_offset = App.Vector(-length / 2 + offset[0], -width / 2 + offset[1], 0)
            global_offset = rotation.multVec(local_offset)

            start_pos = App.Vector(center.x, center.y, center.z) + normal_vec * 0.05 + global_offset
            box.Placement = App.Placement(start_pos, rotation)

            doc.recompute()

            # Boolean cut
            result = obj.Shape.cut(box.Shape)
            cut_obj = doc.addObject("Part::Feature", result_name)
            cut_obj.Shape = result

            doc.removeObject(box_name)
            doc.recompute()

            return {
                "status": "success",
                "object_name": result_name,
                "face": face,
                "width": width,
                "length": length,
                "depth": depth,
            }

        except Exception as e:
            return {"status": "error", "error": f"Failed to create pocket: {e}"}

    # ==================== Smart Placement Commands ====================

    def _cmd_align_to_face(self, params):
        """
        Align one object to a face of another object.

        Useful for placing features relative to existing geometry.

        Parameters:
            object_name: Object to move
            target: Target object
            target_face: Face of target to align to ("top", "front", etc.)
            align_face: Which face of moving object to align (default: "bottom")
            gap: Gap between surfaces (default: 0)

        Example:
            {"command": "align_to_face", "parameters": {
                "object_name": "Widget",
                "target": "Base",
                "target_face": "top",
                "align_face": "bottom",
                "gap": 0.5
            }}
        """
        obj, error = self._get_object(params)
        if error:
            return error

        doc = self._get_doc()
        target_name = params.get("target")
        if not target_name:
            return {"status": "error", "error": "Missing required parameter: 'target'"}

        target = doc.getObject(target_name)
        if not target:
            return {"status": "error", "error": f"Target object '{target_name}' not found"}

        target_face = params.get("target_face", "top")
        align_face = params.get("align_face", "bottom")
        gap = params.get("gap", 0)

        # Get target face info
        target_face_idx = self._get_face_by_direction(target, target_face)
        if not target_face_idx:
            return {"status": "error", "error": f"Could not find target face '{target_face}'"}

        target_face_obj = target.Shape.Faces[target_face_idx - 1]
        target_center = target_face_obj.CenterOfMass
        target_normal = self._get_face_normal(target_face_obj)

        # Get source face info
        source_face_idx = self._get_face_by_direction(obj, align_face)
        if not source_face_idx:
            return {"status": "error", "error": f"Could not find align face '{align_face}'"}

        source_face_obj = obj.Shape.Faces[source_face_idx - 1]
        source_center = source_face_obj.CenterOfMass

        # Calculate movement: align centers, then offset by gap along normal
        normal_vec = App.Vector(target_normal[0], target_normal[1], target_normal[2])
        move_vec = (
            App.Vector(target_center.x, target_center.y, target_center.z)
            - App.Vector(source_center.x, source_center.y, source_center.z)
            + normal_vec * gap
        )

        obj.Placement.Base = obj.Placement.Base + move_vec
        doc.recompute()

        return {
            "status": "success",
            "object_name": obj.Name,
            "aligned_to": target_name,
            "target_face": target_face,
            "moved_by": [move_vec.x, move_vec.y, move_vec.z],
        }

    def _cmd_place_on_face(self, params):
        """
        Place object centered on a face of another object.

        Parameters:
            object_name: Object to place
            target: Target object to place on
            face: Face of target ("top", "front", etc.)
            offset: Optional [x, y] offset on the face plane
            gap: Gap above the surface (default: 0)
        """
        obj, error = self._get_object(params)
        if error:
            return error

        doc = self._get_doc()
        target_name = params.get("target")
        if not target_name:
            return {"status": "error", "error": "Missing required parameter: 'target'"}

        target = doc.getObject(target_name)
        if not target:
            return {"status": "error", "error": f"Target object '{target_name}' not found"}

        face = params.get("face", "top")
        offset = params.get("offset", [0, 0])
        gap = params.get("gap", 0)

        # Get target face
        face_idx = self._get_face_by_direction(target, face)
        if not face_idx:
            return {"status": "error", "error": f"Could not find face '{face}'"}

        face_obj = target.Shape.Faces[face_idx - 1]
        center = face_obj.CenterOfMass
        normal = self._get_face_normal(face_obj)
        normal_vec = App.Vector(normal[0], normal[1], normal[2])

        # Calculate object's bounding box to determine offset for "sitting" on face
        obj_bbox = obj.Shape.BoundBox
        obj_center = obj_bbox.Center

        # Move object so its center is at face center, offset by half height + gap
        half_height = (
            obj_bbox.ZLength / 2
            if normal[2] > 0.5
            else (obj_bbox.YLength / 2 if abs(normal[1]) > 0.5 else obj_bbox.XLength / 2)
        )

        new_pos = App.Vector(center.x + offset[0], center.y + offset[1], center.z) + normal_vec * (half_height + gap)

        # Adjust for current object center offset
        current_base = obj.Placement.Base
        center_offset = App.Vector(obj_center.x, obj_center.y, obj_center.z) - current_base
        obj.Placement.Base = new_pos - center_offset

        doc.recompute()

        return {
            "status": "success",
            "object_name": obj.Name,
            "placed_on": target_name,
            "face": face,
            "position": [obj.Placement.Base.x, obj.Placement.Base.y, obj.Placement.Base.z],
        }

    def _cmd_center_on(self, params):
        """
        Center one object on another in specified axes.

        Parameters:
            object_name: Object to center
            target: Reference object
            axes: List of axes to center on (default: ["x", "y"])
        """
        obj, error = self._get_object(params)
        if error:
            return error

        doc = self._get_doc()
        target_name = params.get("target")
        if not target_name:
            return {"status": "error", "error": "Missing required parameter: 'target'"}

        target = doc.getObject(target_name)
        if not target:
            return {"status": "error", "error": f"Target object '{target_name}' not found"}

        axes = params.get("axes", ["x", "y"])

        obj_center = obj.Shape.BoundBox.Center
        target_center = target.Shape.BoundBox.Center

        move = App.Vector(0, 0, 0)
        if "x" in axes:
            move.x = target_center.x - obj_center.x
        if "y" in axes:
            move.y = target_center.y - obj_center.y
        if "z" in axes:
            move.z = target_center.z - obj_center.z

        obj.Placement.Base = obj.Placement.Base + move
        doc.recompute()

        return {
            "status": "success",
            "object_name": obj.Name,
            "centered_on": target_name,
            "axes": axes,
            "moved_by": [move.x, move.y, move.z],
        }

    # ==================== Analysis & Validation Commands ====================

    def _cmd_check_interference(self, params):
        """
        Check if two objects interfere (overlap/intersect).

        Parameters:
            object_a: First object
            object_b: Second object
            quick: If True (default), use fast bounding-box + distance check only
            calculate_volume: If True, calculate exact interference volume (slow!)

        Returns:
            interferes: Boolean
            distance: Minimum distance (0 if touching, negative if overlapping)
            interference_volume: Volume of intersection (if calculate_volume=True)
        """
        doc = self._get_doc()
        obj_a_name = params.get("object_a") or params.get("object_name")
        obj_b_name = params.get("object_b")
        quick_mode = params.get("quick", True)  # Default to fast mode

        if not obj_a_name or not obj_b_name:
            return {"status": "error", "error": "Both 'object_a' and 'object_b' required"}

        obj_a = doc.getObject(obj_a_name)
        obj_b = doc.getObject(obj_b_name)

        if not obj_a or not obj_b:
            return {"status": "error", "error": "One or both objects not found"}

        if not hasattr(obj_a, "Shape") or not hasattr(obj_b, "Shape"):
            return {"status": "error", "error": "Both objects must have shapes"}

        try:
            # FAST: Bounding box check
            bb_a = obj_a.Shape.BoundBox
            bb_b = obj_b.Shape.BoundBox
            bb_overlap = bb_a.intersect(bb_b)

            # If bounding boxes don't intersect, no interference possible
            if not bb_overlap:
                return {
                    "status": "success",
                    "object_a": obj_a_name,
                    "object_b": obj_b_name,
                    "interferes": False,
                    "distance": 1.0,  # Positive = no overlap
                    "method": "bounding_box",
                }

            # bbox_only mode: just report BB overlap, skip expensive shape checks
            # Useful for quick iterative checks on complex geometry
            if params.get("bbox_only", False):
                return {
                    "status": "success",
                    "object_a": obj_a_name,
                    "object_b": obj_b_name,
                    "interferes": True,  # BBs overlap, assume interference
                    "distance": 0.0,
                    "method": "bounding_box_only",
                }

            # Bounding boxes overlap - need more detailed check
            # Use distToShape which is faster than boolean common
            dist_info = obj_a.Shape.distToShape(obj_b.Shape)
            distance = dist_info[0]

            # In quick mode, use distance as proxy for interference
            # distance <= 0 means shapes are touching or overlapping
            if quick_mode:
                interferes = distance < 0.001  # Small tolerance for touching
                result = {
                    "status": "success",
                    "object_a": obj_a_name,
                    "object_b": obj_b_name,
                    "interferes": interferes,
                    "distance": round(distance, 4),
                    "method": "distance",
                }
                return result

            # SLOW: Full boolean intersection (only if explicitly requested)
            common = obj_a.Shape.common(obj_b.Shape)
            interferes = not common.isNull() and common.Volume > 0.001

            result = {
                "status": "success",
                "object_a": obj_a_name,
                "object_b": obj_b_name,
                "interferes": interferes,
                "distance": round(distance, 4),
                "method": "boolean",
            }

            if params.get("calculate_volume") and interferes:
                result["interference_volume"] = round(common.Volume, 4)

            return result

        except Exception as e:
            return {"status": "error", "error": f"Interference check failed: {e}"}

    def _cmd_analyze_fit(self, params):
        """
        Analyze if one object would fit inside/around another.

        Useful for checking if a phone fits in a holder, etc.

        Parameters:
            inner: Object that should fit inside
            outer: Container/holder object
            clearance: Required clearance on each side (default: 0)

        Returns:
            fits: Boolean
            clearances: {x, y, z} clearance on each axis
            recommendations: Suggestions if it doesn't fit
        """
        doc = self._get_doc()
        inner_name = params.get("inner")
        outer_name = params.get("outer")
        required_clearance = params.get("clearance", 0)

        if not inner_name or not outer_name:
            return {"status": "error", "error": "Both 'inner' and 'outer' required"}

        inner = doc.getObject(inner_name)
        outer = doc.getObject(outer_name)

        if not inner or not outer:
            return {"status": "error", "error": "One or both objects not found"}

        inner_bbox = inner.Shape.BoundBox
        outer_bbox = outer.Shape.BoundBox

        clearances = {
            "x": (outer_bbox.XLength - inner_bbox.XLength) / 2,
            "y": (outer_bbox.YLength - inner_bbox.YLength) / 2,
            "z": (outer_bbox.ZLength - inner_bbox.ZLength) / 2,
        }

        fits = all(c >= required_clearance for c in clearances.values())

        recommendations = []
        if not fits:
            for axis, clearance in clearances.items():
                if clearance < required_clearance:
                    needed = required_clearance - clearance
                    recommendations.append(f"Increase outer {axis.upper()} dimension by {round(needed * 2, 2)}mm")

        return {
            "status": "success",
            "inner": inner_name,
            "outer": outer_name,
            "fits": fits,
            "clearances": {k: round(v, 2) for k, v in clearances.items()},
            "required_clearance": required_clearance,
            "recommendations": recommendations if not fits else None,
        }

    def _cmd_find_face(self, params):
        """
        Find a face dynamically by criteria.

        Parameters:
            object_name: Target object
            direction: Semantic direction ("top", "front", etc.)
            area_gt: Minimum area
            area_lt: Maximum area
            type: Surface type ("Plane", "Cylinder", etc.)
            closest_to: [x, y, z] point to find nearest face

        Returns the best matching face with full details.
        """
        obj, error = self._get_object(params)
        if error:
            return error

        if not hasattr(obj, "Shape"):
            return {"status": "error", "error": f"Object '{obj.Name}' has no shape"}

        shape = obj.Shape
        direction = params.get("direction")
        area_gt = params.get("area_gt")
        area_lt = params.get("area_lt")
        face_type = params.get("type")
        closest_to = params.get("closest_to")

        candidates = []

        for i, f in enumerate(shape.Faces):
            normal = self._get_face_normal(f)

            # Direction filter
            if direction:
                target = self.SEMANTIC_DIRECTIONS.get(direction.lower())
                if target:
                    dot = sum(a * b for a, b in zip(normal, target))
                    if dot < 0.7:  # Allow some tolerance
                        continue

            # Area filter
            if area_gt is not None and f.Area <= area_gt:
                continue
            if area_lt is not None and f.Area >= area_lt:
                continue

            # Type filter
            if face_type and f.Surface.__class__.__name__ != face_type:
                continue

            # Calculate distance if closest_to specified
            dist = 0
            if closest_to:
                pt = App.Vector(closest_to[0], closest_to[1], closest_to[2])
                dist = f.CenterOfMass.distanceToPoint(pt)

            # Determine semantic name
            semantic = None
            for name, dir_vec in self.SEMANTIC_DIRECTIONS.items():
                dot = sum(a * b for a, b in zip(normal, dir_vec))
                if dot > 0.9:
                    semantic = name
                    break

            candidates.append(
                {
                    "index": i + 1,
                    "type": f.Surface.__class__.__name__,
                    "area": round(f.Area, 4),
                    "center": [round(f.CenterOfMass.x, 4), round(f.CenterOfMass.y, 4), round(f.CenterOfMass.z, 4)],
                    "normal": [round(n, 4) for n in normal],
                    "semantic": semantic,
                    "distance": round(dist, 4) if closest_to else None,
                }
            )

        # Sort by relevance
        if closest_to:
            candidates.sort(key=lambda x: x["distance"])
        elif direction:
            # Sort by alignment to direction
            target = self.SEMANTIC_DIRECTIONS.get(direction.lower(), [0, 0, 1])
            candidates.sort(key=lambda x: -sum(a * b for a, b in zip(x["normal"], target)))

        if not candidates:
            return {"status": "success", "object_name": obj.Name, "face": None, "message": "No matching face found"}

        return {
            "status": "success",
            "object_name": obj.Name,
            "face": candidates[0],
            "alternatives": candidates[1:5] if len(candidates) > 1 else [],
        }

    def _cmd_get_face_info(self, params):
        """
        Get detailed info about a specific face.

        Parameters:
            object_name: Target object
            face: Face index (1-based) or semantic name ("top", "front", etc.)

        Returns:
            Detailed face information including edges, area, normal, etc.
        """
        obj, error = self._get_object(params)
        if error:
            return error

        face = params.get("face")
        if not face:
            return {"status": "error", "error": "Missing required parameter: 'face'"}

        # Resolve face reference
        if isinstance(face, str):
            face_idx = self._get_face_by_direction(obj, face)
            if not face_idx:
                return {"status": "error", "error": f"Could not find face '{face}'"}
        else:
            face_idx = face

        if face_idx < 1 or face_idx > len(obj.Shape.Faces):
            return {"status": "error", "error": f"Face index {face_idx} out of range"}

        f = obj.Shape.Faces[face_idx - 1]
        normal = self._get_face_normal(f)
        edge_indices = self._get_face_edge_indices(obj.Shape, f)

        # Get edge details
        edges = []
        for idx in edge_indices:
            e = obj.Shape.Edges[idx - 1]
            edges.append(
                {
                    "index": idx,
                    "type": e.Curve.__class__.__name__,
                    "length": round(e.Length, 4),
                }
            )

        # Determine semantic name
        semantic = None
        for name, dir_vec in self.SEMANTIC_DIRECTIONS.items():
            dot = sum(a * b for a, b in zip(normal, dir_vec))
            if dot > 0.9:
                semantic = name
                break

        return {
            "status": "success",
            "object_name": obj.Name,
            "face": {
                "index": face_idx,
                "semantic": semantic,
                "type": f.Surface.__class__.__name__,
                "area": round(f.Area, 4),
                "center": [round(f.CenterOfMass.x, 4), round(f.CenterOfMass.y, 4), round(f.CenterOfMass.z, 4)],
                "normal": [round(n, 4) for n in normal],
                "edges": edges,
                "edge_count": len(edges),
            },
        }

    # ==================== Export Commands ====================

    def _cmd_export_stl(self, params):
        """Export objects to STL files.

        Supports:
        - Single object or multiple objects
        - ["all"] to export all visible objects
        - combine=True to merge into single STL
        - combine=False to export individual files to directory
        """
        import os

        import Mesh

        doc = self._get_doc()
        # Accept both parameter names for compatibility
        object_names = params.get("object_names") or params.get("objects", [])
        output_path = params.get("output_path") or params.get("filepath")
        # Accept both tolerance and precision
        tolerance = params.get("tolerance") or params.get("precision", 0.05)
        combine = params.get("combine", True)

        # Handle ["all"] - get all visible objects with shapes
        if object_names == ["all"] or object_names == "all":
            object_names = []
            for obj in doc.Objects:
                if hasattr(obj, "Shape") and hasattr(obj, "ViewObject"):
                    if obj.ViewObject and obj.ViewObject.Visibility:
                        object_names.append(obj.Name)

        # Collect shapes - try to export even if isValid() returns False
        # Complex boolean results sometimes report invalid but still tessellate fine
        shapes = []
        shape_names = []
        skipped = []
        for name in object_names:
            obj = doc.getObject(name)
            if not obj:
                skipped.append({"name": name, "reason": "object not found"})
                continue
            if not hasattr(obj, "Shape"):
                skipped.append({"name": name, "reason": "no Shape attribute"})
                continue
            # Try to use shape even if not "valid" - tessellation may still work
            try:
                shape = obj.Shape
                if shape.isNull():
                    skipped.append({"name": name, "reason": "null shape"})
                    continue
                shapes.append(shape)
                shape_names.append(name)
            except Exception as e:
                skipped.append({"name": name, "reason": str(e)})

        if not shapes:
            return {"status": "error", "error": "No valid objects to export", "skipped": skipped}

        exported_files = []
        total_facets = 0
        export_errors = []

        if combine:
            # Combine all shapes into single STL by merging meshes (not boolean fuse - much faster)
            try:
                mesh = Mesh.Mesh()
                for shape in shapes:
                    mesh.addFacets(shape.tessellate(tolerance))
                mesh.write(output_path)
                facet_count = mesh.CountFacets
                total_facets = facet_count
                exported_files.append({"path": output_path, "objects": shape_names, "facets": facet_count})
            except Exception as e:
                return {"status": "error", "error": f"Failed to export combined STL: {str(e)}", "skipped": skipped}
        else:
            # Export individual files to directory
            if not os.path.exists(output_path):
                os.makedirs(output_path)

            for shape, name in zip(shapes, shape_names):
                file_path = os.path.join(output_path, f"{name}.stl")
                try:
                    mesh = Mesh.Mesh()
                    mesh.addFacets(shape.tessellate(tolerance))
                    mesh.write(file_path)
                    facet_count = mesh.CountFacets
                    total_facets += facet_count
                    exported_files.append({"path": file_path, "object": name, "facets": facet_count})
                except Exception as e:
                    export_errors.append({"object": name, "error": str(e)})

        result = {
            "status": "success",
            "files": exported_files,
            "total_files": len(exported_files),
            "total_facets": total_facets,
            "tolerance": tolerance,
        }
        if skipped:
            result["skipped"] = skipped
        if export_errors:
            result["export_errors"] = export_errors
        return result

    def _cmd_export_step(self, params):
        """Export to STEP."""
        doc = self._get_doc()
        objects = params.get("objects", [])
        filepath = params.get("filepath")

        shapes = []
        for name in objects:
            obj = doc.getObject(name)
            if obj and hasattr(obj, "Shape"):
                shapes.append(obj.Shape)

        if shapes:
            combined = shapes[0]
            for shape in shapes[1:]:
                combined = combined.fuse(shape)
            combined.exportStep(filepath)
            return {"status": "success", "filepath": filepath}
        return {"status": "error", "error": "No valid objects to export"}

    # ==================== View Commands ====================

    def _cmd_set_view(self, params):
        """Set view direction."""
        if not HAS_GUI:
            return {"status": "error", "error": "No GUI available"}

        direction = params.get("direction", "isometric")
        view = Gui.ActiveDocument.ActiveView

        view_map = {
            "top": "viewTop",
            "bottom": "viewBottom",
            "front": "viewFront",
            "back": "viewRear",
            "left": "viewLeft",
            "right": "viewRight",
            "isometric": "viewIsometric",
        }

        method = getattr(view, view_map.get(direction.lower(), "viewIsometric"), None)
        if method:
            method()
        view.fitAll()
        return {"status": "success", "view": direction}

    def _cmd_capture_view(self, params):
        """Capture screenshot of current view.

        Parameters:
            output_path/filepath: Path to save the image (PNG format)
            width: Image width in pixels (default 800)
            height: Image height in pixels (default 600)
            view: View angle - "iso", "front", "back", "top", "bottom", "left", "right", "current"
            background: Background color - "white", "black", "gradient", "transparent"
        """
        if not HAS_GUI:
            return {"status": "error", "error": "No GUI available"}

        # Accept both 'output_path' (MCP) and 'filepath' (legacy) parameter names
        filepath = params.get("output_path") or params.get("filepath")
        if not filepath:
            return {"status": "error", "error": "No output path specified (use 'output_path' or 'filepath')"}

        width = params.get("width", 800)
        height = params.get("height", 600)
        view_direction = params.get("view", "current")
        background = params.get("background", "white")

        try:
            active_view = Gui.ActiveDocument.ActiveView

            # Set view direction using direct view methods
            view_methods = {
                "iso": "viewIsometric",
                "isometric": "viewIsometric",
                "front": "viewFront",
                "back": "viewRear",
                "rear": "viewRear",
                "top": "viewTop",
                "bottom": "viewBottom",
                "left": "viewLeft",
                "right": "viewRight",
            }

            view_key = view_direction.lower()
            if view_key in view_methods:
                method_name = view_methods[view_key]
                try:
                    method = getattr(active_view, method_name, None)
                    if method:
                        method()
                        Gui.updateGui()
                except Exception:
                    pass  # View change is optional, continue with capture

            # Fit all objects in view
            try:
                active_view.fitAll()
                Gui.updateGui()
            except Exception:
                pass

            # Map background parameter to FreeCAD saveImage format
            bg_map = {
                "white": "white",
                "black": "black",
                "transparent": "Transparent",
                "gradient": "Current",  # Uses FreeCAD's current gradient
            }
            bg_color = bg_map.get(background.lower(), "white")

            # Capture the image with background color
            active_view.saveImage(str(filepath), int(width), int(height), bg_color)

            return {
                "status": "success",
                "filepath": filepath,
                "width": width,
                "height": height,
                "view": view_direction,
                "background": background,
            }

        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_capture_object(self, params):
        """Capture a single object as an image.

        Isolates the object by hiding others, captures the view, then restores visibility.

        Parameters:
            object_name: Name of object to capture
            output_path: Path to save the image
            width: Image width in pixels (default 600)
            height: Image height in pixels (default 600)
            padding: Zoom padding factor (default 1.2)
            view: View angle - "iso", "front", "top", "right"
            background: Background color - "white", "black", "gradient", "transparent"
            highlight: Apply highlight color to object (default False)
        """
        if not HAS_GUI:
            return {"status": "error", "error": "No GUI available"}

        object_name = params.get("object_name")
        filepath = params.get("output_path") or params.get("filepath")
        if not object_name:
            return {"status": "error", "error": "No object_name specified"}
        if not filepath:
            return {"status": "error", "error": "No output path specified"}

        width = params.get("width", 600)
        height = params.get("height", 600)
        view_direction = params.get("view", "iso")
        background = params.get("background", "white")
        highlight = params.get("highlight", False)

        doc = self._get_doc()
        target_obj = doc.getObject(object_name)
        if not target_obj:
            return {"status": "error", "error": f"Object '{object_name}' not found"}

        original_color = None
        try:
            # Store visibility states and hide all except target
            visibility_states = {}
            for obj in doc.Objects:
                if obj.ViewObject:
                    visibility_states[obj.Name] = obj.ViewObject.Visibility
                    obj.ViewObject.Visibility = obj.Name == object_name

            # Apply highlight color if requested
            if highlight and target_obj.ViewObject:
                original_color = target_obj.ViewObject.ShapeColor
                target_obj.ViewObject.ShapeColor = (0.2, 0.6, 0.9)  # Highlight blue
                Gui.Selection.addSelection(target_obj)

            Gui.updateGui()

            # Get active view and set direction using direct methods
            active_view = Gui.ActiveDocument.ActiveView
            view_methods = {
                "iso": "viewIsometric",
                "isometric": "viewIsometric",
                "front": "viewFront",
                "top": "viewTop",
                "right": "viewRight",
            }

            view_key = view_direction.lower()
            if view_key in view_methods:
                method_name = view_methods[view_key]
                try:
                    method = getattr(active_view, method_name, None)
                    if method:
                        method()
                        Gui.updateGui()
                except Exception:
                    pass

            # Fit to the visible object
            try:
                active_view.fitAll()
                Gui.updateGui()
            except Exception:
                pass

            # Map background parameter to FreeCAD saveImage format
            bg_map = {
                "white": "white",
                "black": "black",
                "transparent": "Transparent",
                "gradient": "Current",
            }
            bg_color = bg_map.get(background.lower(), "white")

            # Capture with background color
            active_view.saveImage(str(filepath), int(width), int(height), bg_color)

            # Restore highlight color if applied
            if highlight and original_color is not None and target_obj.ViewObject:
                target_obj.ViewObject.ShapeColor = original_color
                Gui.Selection.clearSelection()

            # Restore visibility
            for obj_name, was_visible in visibility_states.items():
                obj = doc.getObject(obj_name)
                if obj and obj.ViewObject:
                    obj.ViewObject.Visibility = was_visible

            Gui.updateGui()

            return {
                "status": "success",
                "filepath": filepath,
                "object": object_name,
                "width": width,
                "height": height,
                "background": background,
                "highlighted": highlight,
            }

        except Exception as e:
            # Attempt to restore visibility and color on error
            try:
                if highlight and original_color is not None and target_obj.ViewObject:
                    target_obj.ViewObject.ShapeColor = original_color
                    Gui.Selection.clearSelection()
                for obj_name, was_visible in visibility_states.items():
                    obj = doc.getObject(obj_name)
                    if obj and obj.ViewObject:
                        obj.ViewObject.Visibility = was_visible
            except Exception:
                pass
            return {"status": "error", "error": str(e)}

    # ==================== Script Command ====================

    def _cmd_run_script(self, params):
        """Run arbitrary Python script (escape hatch) with timeout protection."""
        import signal
        import sys

        SCRIPT_TIMEOUT = 10  # seconds

        script = params.get("script", "")

        # Basic input validation
        if not script or not isinstance(script, str):
            return {"status": "error", "error": "Script must be a non-empty string"}

        if len(script) > 50000:
            return {"status": "error", "error": "Script too long (max 50000 chars)"}

        # Block dangerous imports at string level (defense in depth)
        dangerous_patterns = ["import subprocess", "import os", "__import__", "eval(", "compile("]
        for pattern in dangerous_patterns:
            if pattern in script:
                return {"status": "error", "error": f"Blocked pattern detected: {pattern}"}

        # Timeout handler (Unix only - Windows will skip timeout)
        class ScriptTimeoutError(Exception):
            pass

        def timeout_handler(signum, frame):
            raise ScriptTimeoutError("Script execution exceeded 10s limit")

        # Try to set up signal-based timeout (Unix only)
        use_signal_timeout = hasattr(signal, "SIGALRM") and sys.platform != "win32"

        if use_signal_timeout:
            old_handler = signal.signal(signal.SIGALRM, timeout_handler)
            signal.alarm(SCRIPT_TIMEOUT)

        try:
            result = {}
            # Restricted namespace - no builtins that could be dangerous
            safe_builtins = {
                "True": True,
                "False": False,
                "None": None,
                "int": int,
                "float": float,
                "str": str,
                "list": list,
                "dict": dict,
                "len": len,
                "range": range,
                "enumerate": enumerate,
                "zip": zip,
                "min": min,
                "max": max,
                "sum": sum,
                "abs": abs,
                "round": round,
                "print": print,  # Allow print for debugging
            }
            exec_globals = {
                "__builtins__": safe_builtins,
                "App": App,
                "Gui": Gui if HAS_GUI else None,
                "result": result,
                "Part": __import__("Part") if "Part" in sys.modules else None,
            }
            exec(script, exec_globals)
            return {"status": "success", "result": result}
        except ScriptTimeoutError as e:
            return {"status": "error", "error": str(e)}
        except Exception as e:
            return {"status": "error", "error": str(e)}
        finally:
            if use_signal_timeout:
                signal.alarm(0)  # Cancel the alarm
                signal.signal(signal.SIGALRM, old_handler)

    # ==================== Engineering Material Commands ====================

    def _cmd_list_engineering_materials(self, params):
        """List available engineering materials from the server."""
        if not self.materials_client:
            return {"status": "error", "error": "Materials client not available"}

        category = params.get("category")
        try:
            materials = self.materials_client.list_materials(category=category)
            return {
                "status": "success",
                "materials": [m.to_dict() for m in materials],
                "count": len(materials),
            }
        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_get_engineering_material(self, params):
        """Get a specific engineering material by ID."""
        if not self.materials_client:
            return {"status": "error", "error": "Materials client not available"}

        material_id = params.get("material_id")
        if not material_id:
            return {"status": "error", "error": "material_id required"}

        try:
            material = self.materials_client.get_material(material_id)
            if material:
                return {"status": "success", "material": material.to_dict()}
            else:
                return {"status": "error", "error": f"Material not found: {material_id}"}
        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_assign_engineering_material(self, params):
        """Assign an engineering material to a FreeCAD object."""
        if not self.materials_client:
            return {"status": "error", "error": "Materials client not available"}

        object_name = params.get("object_name")
        material_id = params.get("material_id")

        if not object_name or not material_id:
            return {"status": "error", "error": "object_name and material_id required"}

        doc = self._get_doc()
        obj = doc.getObject(object_name)
        if not obj:
            return {"status": "error", "error": f"Object not found: {object_name}"}

        # Get the material
        material = self.materials_client.get_material(material_id)
        if not material:
            return {"status": "error", "error": f"Material not found: {material_id}"}

        # Store assignment in materials client
        self.materials_client.assign_material(object_name, material_id)

        # Store material info as custom properties on the FreeCAD object
        # FreeCAD objects support dynamic properties
        try:
            # Add custom properties if they don't exist
            if not hasattr(obj, "ConjureMaterialId"):
                obj.addProperty("App::PropertyString", "ConjureMaterialId", "Conjure", "Engineering material ID")
            if not hasattr(obj, "ConjureMaterialName"):
                obj.addProperty("App::PropertyString", "ConjureMaterialName", "Conjure", "Engineering material name")
            if not hasattr(obj, "ConjureDensity"):
                obj.addProperty("App::PropertyFloat", "ConjureDensity", "Conjure", "Density (kg/m³)")
            if not hasattr(obj, "ConjureYoungsModulus"):
                obj.addProperty("App::PropertyFloat", "ConjureYoungsModulus", "Conjure", "Young's modulus (Pa)")

            # Set property values
            obj.ConjureMaterialId = material.id
            obj.ConjureMaterialName = material.name
            if material.density_kg_m3:
                obj.ConjureDensity = material.density_kg_m3
            if material.youngs_modulus_pa:
                obj.ConjureYoungsModulus = material.youngs_modulus_pa

            doc.recompute()

            # Apply visual material if GUI available
            if HAS_GUI and material.base_color:
                try:
                    view_obj = obj.ViewObject
                    if view_obj:
                        r, g, b = material.base_color
                        view_obj.ShapeColor = (r, g, b)
                except Exception:
                    pass  # Visual update is optional

        except Exception as e:
            return {"status": "error", "error": f"Failed to set properties: {e}"}

        return {
            "status": "success",
            "object": object_name,
            "material_id": material_id,
            "material_name": material.name,
            "density_kg_m3": material.density_kg_m3,
        }

    def _cmd_get_object_engineering_material(self, params):
        """Get the engineering material assigned to an object."""
        object_name = params.get("object_name")
        if not object_name:
            return {"status": "error", "error": "object_name required"}

        doc = self._get_doc()
        obj = doc.getObject(object_name)
        if not obj:
            return {"status": "error", "error": f"Object not found: {object_name}"}

        # Check for material properties on the object
        if hasattr(obj, "ConjureMaterialId") and obj.ConjureMaterialId:
            material_id = obj.ConjureMaterialId
            material_name = getattr(obj, "ConjureMaterialName", "")
            density = getattr(obj, "ConjureDensity", None)
            youngs_modulus = getattr(obj, "ConjureYoungsModulus", None)

            # Try to get full material from cache
            if self.materials_client:
                material = self.materials_client.get_material(material_id)
                if material:
                    return {
                        "status": "success",
                        "object": object_name,
                        "material": material.to_dict(),
                    }

            # Fallback to stored properties
            return {
                "status": "success",
                "object": object_name,
                "material": {
                    "id": material_id,
                    "name": material_name,
                    "density_kg_m3": density,
                    "youngs_modulus_pa": youngs_modulus,
                },
            }
        else:
            return {"status": "success", "object": object_name, "material": None}

    def _cmd_clear_engineering_material(self, params):
        """Clear the engineering material assignment from an object."""
        object_name = params.get("object_name")
        if not object_name:
            return {"status": "error", "error": "object_name required"}

        doc = self._get_doc()
        obj = doc.getObject(object_name)
        if not obj:
            return {"status": "error", "error": f"Object not found: {object_name}"}

        # Clear from materials client
        if self.materials_client:
            self.materials_client.clear_object_material(object_name)

        # Clear custom properties
        try:
            if hasattr(obj, "ConjureMaterialId"):
                obj.ConjureMaterialId = ""
            if hasattr(obj, "ConjureMaterialName"):
                obj.ConjureMaterialName = ""
            if hasattr(obj, "ConjureDensity"):
                obj.ConjureDensity = 0.0
            if hasattr(obj, "ConjureYoungsModulus"):
                obj.ConjureYoungsModulus = 0.0

            # Reset color to default
            if HAS_GUI:
                try:
                    view_obj = obj.ViewObject
                    if view_obj:
                        view_obj.ShapeColor = (0.8, 0.8, 0.8)  # Default gray
                except Exception:
                    pass

            doc.recompute()
        except Exception as e:
            return {"status": "error", "error": f"Failed to clear properties: {e}"}

        return {"status": "success", "object": object_name, "cleared": True}

    def _cmd_refresh_materials_cache(self, params):
        """Force refresh of the materials cache."""
        if not self.materials_client:
            return {"status": "error", "error": "Materials client not available"}

        try:
            success = self.materials_client.refresh_cache()
            if success:
                materials = self.materials_client.list_materials()
                return {
                    "status": "success",
                    "refreshed": True,
                    "materials_count": len(materials),
                }
            else:
                return {"status": "error", "error": "Failed to refresh cache"}
        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_get_materials_categories(self, params):
        """Get list of material categories."""
        if not self.materials_client:
            return {"status": "error", "error": "Materials client not available"}

        try:
            categories = self.materials_client.get_categories()
            return {"status": "success", "categories": categories}
        except Exception as e:
            return {"status": "error", "error": str(e)}

    # ==================== Standards Library ====================

    def _cmd_list_standards(self, params):
        """
        List available standards in the library.

        Args:
            category: Optional filter - "socket", "fastener", "material",
                     "thread", "profile", or "gear"

        Returns:
            Dictionary of category -> list of standard IDs
        """
        if not HAS_STANDARDS:
            return {"status": "error", "error": "Standards library not available"}

        lib = get_standards_library()
        if not lib:
            return {"status": "error", "error": "Standards library not available"}

        try:
            category = params.get("category")
            result = lib.list_standards(category)
            return {"status": "success", "standards": result}
        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_get_standard(self, params):
        """
        Get a specific standard specification.

        Args:
            category: Standard category ("socket", "fastener", "material",
                     "thread", "profile", "gear")
            spec_id: Standard ID (e.g., "hex_8mm", "spur_m1_z20")

        Returns:
            Specification dictionary with all dimensions
        """
        if not HAS_STANDARDS:
            return {"status": "error", "error": "Standards library not available"}

        lib = get_standards_library()
        if not lib:
            return {"status": "error", "error": "Standards library not available"}

        category = params.get("category")
        spec_id = params.get("spec_id")

        if not category:
            return {"status": "error", "error": "category is required"}
        if not spec_id:
            return {"status": "error", "error": "spec_id is required"}

        try:
            spec = lib.get_standard(category, spec_id)
            if not spec:
                return {"status": "error", "error": f"Standard '{spec_id}' not found in {category}"}
            return {"status": "success", "spec": spec}
        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_search_standards(self, params):
        """
        Search for standards matching a query.

        Args:
            query: Search term (matches ID, description, or use cases)
            category: Optional category filter

        Returns:
            List of matching specifications
        """
        if not HAS_STANDARDS:
            return {"status": "error", "error": "Standards library not available"}

        lib = get_standards_library()
        if not lib:
            return {"status": "error", "error": "Standards library not available"}

        query = params.get("query")
        if not query:
            return {"status": "error", "error": "query is required"}

        category = params.get("category")

        try:
            results = lib.search_standards(query, category)
            return {"status": "success", "results": results, "count": len(results)}
        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_get_gear(self, params):
        """
        Get gear specification with calculated involute geometry.

        Args:
            gear_id: Gear ID (e.g., "spur_m1_z20", "planetary_sun_m1_z12")

        Returns:
            Complete gear specification including:
            - module_mm, num_teeth, pressure_angle_deg
            - pitch_diameter_mm, base_diameter_mm
            - addendum_mm, dedendum_mm
            - tip_diameter_mm, root_diameter_mm
            - tooth_thickness_mm, root_fillet_radius_mm
        """
        if not HAS_STANDARDS:
            return {"status": "error", "error": "Standards library not available"}

        lib = get_standards_library()
        if not lib:
            return {"status": "error", "error": "Standards library not available"}

        gear_id = params.get("gear_id")
        if not gear_id:
            return {"status": "error", "error": "gear_id is required"}

        try:
            gear = lib.get_gear(gear_id)
            if not gear:
                available = lib.list_gears()[:10]
                return {
                    "status": "error",
                    "error": f"Gear '{gear_id}' not found",
                    "available_samples": available,
                }
            return {"status": "success", "gear": gear}
        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_list_gears(self, params):
        """
        List all available gear specifications.

        Returns:
            List of gear IDs
        """
        if not HAS_STANDARDS:
            return {"status": "error", "error": "Standards library not available"}

        lib = get_standards_library()
        if not lib:
            return {"status": "error", "error": "Standards library not available"}

        try:
            gears = lib.list_gears()
            return {"status": "success", "gears": gears, "count": len(gears)}
        except Exception as e:
            return {"status": "error", "error": str(e)}

    def _cmd_get_gear_formulas(self, params):
        """
        Get gear geometry formulas for reference.

        Returns:
            Dictionary of involute gear formulas and design guidelines
        """
        if not HAS_STANDARDS:
            return {"status": "error", "error": "Standards library not available"}

        lib = get_standards_library()
        if not lib:
            return {"status": "error", "error": "Standards library not available"}

        try:
            formulas = lib.get_formulas("gears")
            return {"status": "success", "formulas": formulas}
        except Exception as e:
            return {"status": "error", "error": str(e)}

    # ==================== Simulation ====================

    def _cmd_calculate_dynamic_properties(self, params):
        """
        Calculate dynamic properties (mass, volume, CoM, inertia) for an object.

        Uses FreeCAD's Part module for precise B-rep calculations.
        """
        obj_name = params.get("object")
        material_id = params.get("material_id")
        density = params.get("density")

        if not obj_name:
            return {"status": "error", "error": "object is required"}

        doc = self._get_doc()
        obj = doc.getObject(obj_name)
        if not obj:
            return {"status": "error", "error": f"Object '{obj_name}' not found"}

        if not hasattr(obj, "Shape"):
            return {"status": "error", "error": f"Object '{obj_name}' has no Shape"}

        shape = obj.Shape

        # Get density from material library or parameter
        if density is None:
            if material_id and self.materials_client:
                material = self.materials_client.get_material(material_id)
                if material and material.density_kg_m3:
                    density = material.density_kg_m3
                else:
                    density = 7850  # Default steel
            else:
                density = 7850  # Default steel

        try:
            # FreeCAD volumes are in mm³, convert to m³
            volume_mm3 = shape.Volume
            volume_m3 = volume_mm3 * 1e-9  # mm³ to m³

            # Mass in kg
            mass_kg = volume_m3 * density

            # Surface area (mm² to m²)
            surface_area_m2 = shape.Area * 1e-6

            # Center of mass (FreeCAD uses mm, convert to m)
            com = shape.CenterOfMass
            center_of_mass_m = [com.x * 0.001, com.y * 0.001, com.z * 0.001]

            # Moments of inertia - FreeCAD provides MatrixOfInertia
            # This is the inertia tensor at the center of mass
            # Units: g·mm² - need to convert to kg·m²
            inertia_matrix = shape.MatrixOfInertia

            # Extract diagonal and off-diagonal elements
            # MatrixOfInertia is a 4x4 matrix, upper 3x3 is the inertia tensor
            # Convert from g·mm² to kg·m² (multiply by 1e-9)
            conversion = density / 1000 * 1e-9  # g/mm³ to kg/m³ * mm² to m²

            moments_of_inertia = {
                "Ixx": inertia_matrix.A11 * conversion,
                "Iyy": inertia_matrix.A22 * conversion,
                "Izz": inertia_matrix.A33 * conversion,
                "Ixy": inertia_matrix.A12 * conversion,
                "Ixz": inertia_matrix.A13 * conversion,
                "Iyz": inertia_matrix.A23 * conversion,
            }

            # Bounding box (mm to m)
            bbox = shape.BoundBox
            bounding_box = {
                "min": [bbox.XMin * 0.001, bbox.YMin * 0.001, bbox.ZMin * 0.001],
                "max": [bbox.XMax * 0.001, bbox.YMax * 0.001, bbox.ZMax * 0.001],
                "size": [
                    (bbox.XMax - bbox.XMin) * 0.001,
                    (bbox.YMax - bbox.YMin) * 0.001,
                    (bbox.ZMax - bbox.ZMin) * 0.001,
                ],
                "center": [
                    (bbox.XMin + bbox.XMax) * 0.0005,
                    (bbox.YMin + bbox.YMax) * 0.0005,
                    (bbox.ZMin + bbox.ZMax) * 0.0005,
                ],
            }

            return {
                "status": "success",
                "object": obj_name,
                "dynamic_properties": {
                    "mass_kg": mass_kg,
                    "volume_m3": volume_m3,
                    "surface_area_m2": surface_area_m2,
                    "center_of_mass_m": center_of_mass_m,
                    "moments_of_inertia_kg_m2": moments_of_inertia,
                    "bounding_box_m": bounding_box,
                    "density_kg_m3": density,
                },
            }

        except Exception as e:
            return {"status": "error", "error": f"Calculation failed: {str(e)}"}

    def _cmd_get_simulation_capabilities(self, params):
        """Get the simulation capabilities of this FreeCAD client."""
        return {
            "status": "success",
            "capabilities": {
                "dynamic_properties": {
                    "supported": True,
                    "mass_calculation": True,
                    "volume_calculation": True,
                    "center_of_mass": True,
                    "moments_of_inertia": True,
                    "surface_area": True,
                    "precision": "high",  # B-rep based
                },
                "stress_analysis": {
                    "supported": False,
                    "note": "Use FEM workbench or server-side BeamEstimator",
                },
                "thermal_analysis": {
                    "supported": False,
                    "note": "Use FEM workbench or server-side ThermalEstimator",
                },
                "physics_simulation": {
                    "supported": False,
                    "note": "FreeCAD does not have real-time physics",
                },
            },
            "geometry_kernel": "OpenCASCADE",
            "precision": "double",
        }

    def _cmd_export_geometry_ugf(self, params):
        """
        Export object geometry in Universal Geometry Format (UGF).

        UGF is a mesh-based format for geometry interchange between Conjure clients.
        """
        obj_name = params.get("object")
        include_materials = params.get("include_materials", False)
        mesh_quality = params.get("mesh_quality", 0.1)  # Linear deflection

        if not obj_name:
            return {"status": "error", "error": "object is required"}

        doc = self._get_doc()
        obj = doc.getObject(obj_name)
        if not obj:
            return {"status": "error", "error": f"Object '{obj_name}' not found"}

        if not hasattr(obj, "Shape"):
            return {"status": "error", "error": f"Object '{obj_name}' has no Shape"}

        try:
            import Part

            shape = obj.Shape

            # Tessellate the shape
            mesh_data = shape.tessellate(mesh_quality)
            vertices = mesh_data[0]  # List of Vector
            faces = mesh_data[1]  # List of tuples (v1, v2, v3)

            # Convert to lists (mm to m)
            vertices_list = [[v.x * 0.001, v.y * 0.001, v.z * 0.001] for v in vertices]
            faces_list = [list(f) for f in faces]

            # Compute normals for each vertex
            normals_list = []
            for _v in vertices:
                # Approximate normal at vertex by averaging face normals
                # For now, use Z-up as default (would need proper normal calculation)
                normals_list.append([0.0, 0.0, 1.0])

            # Get transform
            placement = obj.Placement
            location = [
                placement.Base.x * 0.001,
                placement.Base.y * 0.001,
                placement.Base.z * 0.001,
            ]
            # Rotation as Euler angles (degrees)
            yaw, pitch, roll = placement.Rotation.toEuler()

            ugf = {
                "format": "UGF",
                "version": "1.0",
                "source": "FreeCAD",
                "object_name": obj_name,
                "vertices": vertices_list,
                "faces": faces_list,
                "normals": normals_list,
                "transform": {
                    "location": location,
                    "rotation": [roll, pitch, yaw],  # XYZ Euler
                    "scale": [1.0, 1.0, 1.0],
                },
            }

            if include_materials:
                # Check for engineering material assignment
                mat_id = obj.ViewObject.ShapeColor if HAS_GUI else None
                if hasattr(obj, "conjure_material_id"):
                    mat_id = obj.conjure_material_id
                    ugf["material_id"] = mat_id

            return {
                "status": "success",
                "object": obj_name,
                "ugf": ugf,
                "vertex_count": len(vertices_list),
                "face_count": len(faces_list),
            }

        except Exception as e:
            return {"status": "error", "error": f"Export failed: {str(e)}"}

    def _cmd_import_geometry_ugf(self, params):
        """
        Import geometry from Universal Geometry Format (UGF).

        Creates a mesh object from the UGF data.
        """
        ugf_data = params.get("ugf")
        object_name = params.get("name")

        if not ugf_data:
            return {"status": "error", "error": "ugf data is required"}

        if ugf_data.get("format") != "UGF":
            return {"status": "error", "error": "Invalid UGF format"}

        try:
            import Mesh
            import Part

            vertices = ugf_data.get("vertices", [])
            faces = ugf_data.get("faces", [])
            name = object_name or ugf_data.get("object_name", "ImportedMesh")

            if not vertices or not faces:
                return {"status": "error", "error": "UGF has no geometry data"}

            doc = self._get_doc()

            # Convert vertices from m to mm
            vertices_mm = [(v[0] * 1000, v[1] * 1000, v[2] * 1000) for v in vertices]

            # Create mesh
            mesh_obj = doc.addObject("Mesh::Feature", name)
            mesh = Mesh.Mesh()

            for face in faces:
                if len(face) >= 3:
                    v1 = vertices_mm[face[0]]
                    v2 = vertices_mm[face[1]]
                    v3 = vertices_mm[face[2]]
                    mesh.addFacet(v1, v2, v3)

            mesh_obj.Mesh = mesh

            # Apply transform if provided
            if "transform" in ugf_data:
                t = ugf_data["transform"]
                if "location" in t:
                    loc = t["location"]
                    mesh_obj.Placement.Base = App.Vector(loc[0] * 1000, loc[1] * 1000, loc[2] * 1000)
                if "rotation" in t:
                    rot = t["rotation"]
                    mesh_obj.Placement.Rotation = App.Rotation(rot[2], rot[1], rot[0])

            doc.recompute()

            return {
                "status": "success",
                "object": mesh_obj.Name,
                "vertex_count": len(vertices),
                "face_count": len(faces),
            }

        except Exception as e:
            return {"status": "error", "error": f"Import failed: {str(e)}"}

    def _cmd_get_adapter_capabilities(self, params):
        """Get full adapter capabilities for server registration."""
        import platform

        return {
            "status": "success",
            "adapter_type": "freecad",
            "adapter_id": f"freecad_{id(self)}",
            "version": "1.0.0",
            "freecad_version": App.Version()[0] + "." + App.Version()[1],
            "capabilities": {
                "cad_operations": [
                    "primitives",
                    "booleans",
                    "transforms",
                    "modifiers",
                    "queries",
                    "export",
                    "topology",
                    "faces",
                    "measurements",
                ],
                "simulation": {
                    "dynamic_properties": {
                        "supported": True,
                        "mass_calculation": True,
                        "volume_calculation": True,
                        "center_of_mass": True,
                        "moments_of_inertia": True,
                        "surface_area": True,
                    },
                    "stress_analysis": {
                        "supported": False,
                        "note": "Use server-side BeamEstimator",
                    },
                    "thermal_analysis": {
                        "supported": False,
                        "note": "Use server-side ThermalEstimator",
                    },
                    "physics": {
                        "supported": False,
                        "note": "Use Blender client for physics simulations",
                    },
                },
                "geometry_kernel": "OpenCASCADE",
                "export_formats": ["STEP", "STL", "IGES", "BREP"],
                "import_formats": ["STEP", "STL", "IGES", "BREP", "OBJ"],
                "streaming_results": True,
            },
            "resource_limits": {
                "max_vertices": 50_000_000,
                "max_objects": 10_000,
            },
            "system_info": {
                "os": platform.system(),
                "python_version": platform.python_version(),
            },
        }

    # ==================== Helpers ====================

    def _get_doc(self):
        """Get or create active document."""
        doc = App.ActiveDocument
        if not doc:
            doc = App.newDocument("Untitled")
        return doc


# Global server instance
_server = None


def start_server():
    """Start the Conjure server."""
    global _server
    if _server is None:
        _server = ConjureServer()
    _server.start()
    return _server


def stop_server():
    """Stop the Conjure server."""
    global _server
    if _server:
        _server.stop()
        _server = None


def get_server():
    """Get the server instance."""
    return _server


# ============================================================================
# CLOUD BRIDGE - WebSocket connection to hosted Conjure server
# ============================================================================

import base64
import hashlib
import os
import ssl
import struct
from urllib.parse import urlparse


class SimpleWebSocket:
    """
    Minimal WebSocket client using only Python stdlib.
    Supports wss:// (TLS) connections for secure communication.
    """

    GUID = "258EAFA5-E914-47DA-95CA-C5AB0DC85B11"

    def __init__(self, url, headers=None):
        self.url = url
        self.headers = headers or {}
        self.sock = None
        self.connected = False

    def connect(self):
        """Establish WebSocket connection."""
        parsed = urlparse(self.url)
        host = parsed.hostname
        port = parsed.port or (443 if parsed.scheme == "wss" else 80)
        path = parsed.path or "/"
        if parsed.query:
            path += "?" + parsed.query

        # Create socket
        raw_sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        raw_sock.settimeout(10)

        if parsed.scheme == "wss":
            context = ssl.create_default_context()
            self.sock = context.wrap_socket(raw_sock, server_hostname=host)
        else:
            self.sock = raw_sock

        self.sock.connect((host, port))

        # WebSocket handshake
        key = base64.b64encode(os.urandom(16)).decode()
        handshake = f"GET {path} HTTP/1.1\r\n"
        handshake += f"Host: {host}\r\n"
        handshake += "Upgrade: websocket\r\n"
        handshake += "Connection: Upgrade\r\n"
        handshake += f"Sec-WebSocket-Key: {key}\r\n"
        handshake += "Sec-WebSocket-Version: 13\r\n"
        for k, v in self.headers.items():
            handshake += f"{k}: {v}\r\n"
        handshake += "\r\n"

        self.sock.sendall(handshake.encode())

        # Read response
        response = b""
        while b"\r\n\r\n" not in response:
            chunk = self.sock.recv(1024)
            if not chunk:
                raise ConnectionError("Connection closed during handshake")
            response += chunk

        # Verify upgrade
        if b"101" not in response.split(b"\r\n")[0]:
            raise ConnectionError(f"WebSocket upgrade failed: {response[:100]}")

        # Verify accept key
        expected = base64.b64encode(hashlib.sha1((key + self.GUID).encode()).digest()).decode()
        if expected.encode() not in response:
            raise ConnectionError("Invalid Sec-WebSocket-Accept")

        self.connected = True

    def send(self, data):
        """Send text frame."""
        if isinstance(data, str):
            data = data.encode()

        # Build frame: FIN + text opcode
        frame = bytearray([0x81])  # FIN=1, opcode=1 (text)

        # Length + mask bit
        length = len(data)
        if length < 126:
            frame.append(0x80 | length)  # Mask bit set
        elif length < 65536:
            frame.append(0x80 | 126)
            frame.extend(struct.pack(">H", length))
        else:
            frame.append(0x80 | 127)
            frame.extend(struct.pack(">Q", length))

        # Mask key and masked data
        mask = os.urandom(4)
        frame.extend(mask)
        for i, byte in enumerate(data):
            frame.append(byte ^ mask[i % 4])

        self.sock.sendall(frame)

    def recv(self):
        """Receive text frame. Returns string or None on close."""
        # Read frame header
        header = self._recv_exact(2)
        if not header:
            return None

        # fin = header[0] & 0x80  # FIN bit (unused, we don't handle fragmentation)
        opcode = header[0] & 0x0F
        masked = header[1] & 0x80
        length = header[1] & 0x7F

        # Extended length
        if length == 126:
            length = struct.unpack(">H", self._recv_exact(2))[0]
        elif length == 127:
            length = struct.unpack(">Q", self._recv_exact(8))[0]

        # Mask key (if server sends masked, which it shouldn't)
        mask = self._recv_exact(4) if masked else None

        # Payload
        data = self._recv_exact(length)
        if mask:
            data = bytes(b ^ mask[i % 4] for i, b in enumerate(data))

        # Handle opcodes
        if opcode == 0x08:  # Close
            return None
        elif opcode == 0x09:  # Ping
            self._send_pong(data)
            return self.recv()
        elif opcode == 0x0A:  # Pong
            return self.recv()
        elif opcode in (0x01, 0x02):  # Text or binary
            return data.decode() if opcode == 0x01 else data

        return None

    def _recv_exact(self, n):
        """Receive exactly n bytes."""
        data = b""
        while len(data) < n:
            chunk = self.sock.recv(n - len(data))
            if not chunk:
                raise ConnectionError("Connection closed")
            data += chunk
        return data

    def _send_pong(self, data):
        """Send pong frame."""
        frame = bytearray([0x8A, 0x80 | len(data)])
        mask = os.urandom(4)
        frame.extend(mask)
        for i, byte in enumerate(data):
            frame.append(byte ^ mask[i % 4])
        self.sock.sendall(frame)

    def close(self):
        """Close connection."""
        if self.sock:
            try:
                # Send close frame
                self.sock.sendall(bytes([0x88, 0x80, 0, 0, 0, 0]))
            except Exception:
                pass
            try:
                self.sock.close()
            except Exception:
                pass
        self.sock = None
        self.connected = False

    def settimeout(self, timeout):
        """Set socket timeout."""
        if self.sock:
            self.sock.settimeout(timeout)


class CloudBridge:
    """
    WebSocket bridge to Conjure Cloud server.

    Connects to the cloud server, receives commands, and executes them locally.
    This enables AI assistants (Claude, Cursor, etc.) to control FreeCAD.
    Uses only Python stdlib - no external dependencies required.
    """

    def __init__(self, server: ConjureServer, api_key: str, server_url: str = None):
        self.server = server
        self.api_key = api_key
        self.server_url = server_url or "wss://conjure.lautrek.com/api/v1/adapter/ws"
        self.ws = None
        self.running = False
        self.thread = None
        self.adapter_id = None
        self._heartbeat_thread = None
        self._ws_lock = threading.Lock()  # Protect WebSocket send

    def start(self):
        """Start cloud bridge in background thread."""
        if self.running:
            return

        self.running = True
        self.thread = threading.Thread(target=self._run_loop, daemon=True)
        self.thread.start()
        App.Console.PrintMessage(f"Cloud bridge connecting to {self.server_url}\n")

    def stop(self):
        """Stop cloud bridge."""
        self.running = False
        if self.ws:
            try:
                self.ws.close()
            except Exception:
                pass
        self.ws = None
        self.adapter_id = None
        App.Console.PrintMessage("Cloud bridge stopped\n")

    def _run_loop(self):
        """Main connection loop with reconnection."""
        import time

        while self.running:
            try:
                self._connect_and_run()
            except Exception as e:
                App.Console.PrintWarning(f"Cloud bridge error: {e}\n")
                self.adapter_id = None

            if self.running:
                # Wait before reconnecting
                time.sleep(5)

    def _connect_and_run(self):
        """Connect and process messages."""
        # Create WebSocket connection with auth header
        self.ws = SimpleWebSocket(
            self.server_url,
            headers={"Authorization": f"Bearer {self.api_key}"},
        )
        self.ws.connect()

        App.Console.PrintMessage("Cloud bridge: WebSocket connected\n")

        # Send registration with full capabilities
        registration = {
            "type": "adapter_registration",
            "adapter_type": "freecad",
            "adapter_id": f"freecad_{id(self.server)}",
            "version": "1.0.0",
            "freecad_version": App.Version()[0] + "." + App.Version()[1],
            "capabilities": {
                "cad_operations": [
                    "primitives",
                    "booleans",
                    "transforms",
                    "modifiers",
                    "queries",
                    "export",
                    "topology",
                    "faces",
                    "measurements",
                ],
                "simulation": {
                    "dynamic_properties": {
                        "supported": True,
                        "mass_calculation": True,
                        "volume_calculation": True,
                        "center_of_mass": True,
                        "moments_of_inertia": True,
                        "surface_area": True,
                    },
                    "stress_analysis": {"supported": False},
                    "thermal_analysis": {"supported": False},
                    "physics": {"supported": False},
                },
                "geometry_kernel": "OpenCASCADE",
                "export_formats": ["STEP", "STL", "IGES", "BREP"],
                "import_formats": ["STEP", "STL", "IGES", "BREP", "OBJ"],
            },
        }
        self.ws.send(json.dumps(registration))

        # Wait for registration confirmation
        resp = json.loads(self.ws.recv())
        if resp.get("type") == "registration_confirmed":
            self.adapter_id = resp.get("adapter_id")
            App.Console.PrintMessage(f"Cloud bridge: Registered as {self.adapter_id}\n")
        else:
            App.Console.PrintWarning(f"Cloud bridge: Unexpected response: {resp}\n")
            return

        # Start proactive heartbeat thread
        self._heartbeat_thread = threading.Thread(target=self._heartbeat_loop, daemon=True)
        self._heartbeat_thread.start()

        # Message loop
        while self.running:
            try:
                self.ws.settimeout(60)  # Longer timeout, heartbeats are proactive now
                raw = self.ws.recv()
                if raw is None:
                    # Connection closed
                    App.Console.PrintWarning("Cloud bridge: Connection closed\n")
                    break
                message = json.loads(raw)
                self._handle_message(message)
            except socket.timeout:
                # Timeout just means no messages, heartbeat thread handles keepalive
                pass
            except (ConnectionError, OSError) as e:
                App.Console.PrintWarning(f"Cloud bridge: Connection error: {e}\n")
                break
            except Exception as e:
                App.Console.PrintWarning(f"Cloud bridge message error: {e}\n")
                break

    def _handle_message(self, message):
        """Handle incoming message from cloud."""
        msg_type = message.get("type")

        if msg_type == "execute_command":
            self._handle_execute_command(message)
        elif msg_type == "heartbeat_ack":
            pass  # Heartbeat acknowledged
        else:
            App.Console.PrintWarning(f"Cloud bridge: Unknown message type: {msg_type}\n")

    def _handle_execute_command(self, message):
        """Execute command and send result back."""
        request_id = message.get("request_id")
        command_type = message.get("command_type")
        params = message.get("params", {})

        App.Console.PrintMessage(f"Cloud bridge: Executing {command_type}\n")

        # Execute via thread-safe method (queues to main thread)
        try:
            result_data = self.server.execute_threadsafe(command_type, params, timeout=30.0)

            # Send result back
            response = {
                "type": "command_result",
                "request_id": request_id,
                "success": result_data.get("status") == "success",
                "data": result_data,
                "error": result_data.get("error"),
            }
            with self._ws_lock:
                self.ws.send(json.dumps(response))

        except Exception as e:
            # Send error back
            response = {
                "type": "command_result",
                "request_id": request_id,
                "success": False,
                "data": {},
                "error": str(e),
            }
            with self._ws_lock:
                self.ws.send(json.dumps(response))

    def _heartbeat_loop(self):
        """Proactive heartbeat sender - runs in separate thread."""
        import time

        while self.running and self.ws:
            time.sleep(25)  # Send heartbeat every 25 seconds (server timeout is 60s)
            if not self.running or not self.ws:
                break
            try:
                with self._ws_lock:
                    self.ws.send(json.dumps({"type": "heartbeat"}))
            except Exception:
                # Connection issue, main loop will handle reconnect
                break


# Global cloud bridge instance
_cloud_bridge = None


def start_cloud_bridge(api_key: str, server_url: str = None):
    """Start the cloud bridge."""
    global _cloud_bridge, _server

    if _server is None:
        start_server()

    # Derive HTTP API URL from WebSocket URL and update materials client
    if server_url and _server and MaterialsClient:
        try:
            # Convert wss://host/path to https://host
            api_url = server_url.replace("wss://", "https://").replace("ws://", "http://")
            # Remove the WebSocket path to get base URL
            if "/api/" in api_url:
                api_url = api_url.split("/api/")[0]
            elif "/ws" in api_url:
                api_url = api_url.split("/ws")[0]

            # Update or create materials client with correct URL
            _server.materials_client = MaterialsClient(api_url, api_key)
            App.Console.PrintMessage(f"Conjure: Materials API URL set to {api_url}\n")
        except Exception as e:
            App.Console.PrintWarning(f"Conjure: Failed to configure materials client: {e}\n")

    if _cloud_bridge is None:
        _cloud_bridge = CloudBridge(_server, api_key, server_url)

    _cloud_bridge.start()

    # Hook up placement observer for constraint propagation
    observer = get_placement_observer()
    observer.set_cloud_bridge(_cloud_bridge)

    return _cloud_bridge


def stop_cloud_bridge():
    """Stop the cloud bridge."""
    global _cloud_bridge
    if _cloud_bridge:
        _cloud_bridge.stop()
        _cloud_bridge = None


def get_cloud_bridge():
    """Get the cloud bridge instance."""
    return _cloud_bridge
