import builtins
import contextlib
import os

import FreeCAD as App
import FreeCADGui as Gui

# Global panel reference
mcp_panel = None


# SimpleMCPPanel defined in Activated method to avoid FreeCAD module reloading issues


class ConjureShowCommand:
    """Command to show the Conjure panel"""

    def GetResources(self):
        user_dir = App.getUserAppDataDir()
        icon_path = os.path.join(user_dir, "Mod", "conjure", "assets", "conjure_icon.svg")
        return {
            "Pixmap": icon_path,
            "MenuText": "Show Conjure Panel",
            "ToolTip": "Show the Conjure - AI CAD Control panel",
        }

    def IsActive(self):
        return True

    def Activated(self):
        global mcp_panel
        print("INFO:Conjure_Gui:Opening Conjure Dashboard...")

        try:
            import json
            import socket
            import threading
            import time

            from PySide2 import QtCore, QtGui, QtWidgets

            class MCPDashboard:
                """Comprehensive MCP Server Dashboard with monitoring and control"""

                # Modern color scheme
                COLORS = {
                    "primary": "#2563eb",  # Blue
                    "primary_dark": "#1d4ed8",
                    "success": "#10b981",  # Green
                    "warning": "#f59e0b",  # Amber
                    "danger": "#ef4444",  # Red
                    "bg_dark": "#1e293b",  # Slate 800
                    "bg_card": "#334155",  # Slate 700
                    "bg_light": "#f8fafc",  # Slate 50
                    "text_primary": "#f8fafc",  # Light text
                    "text_secondary": "#94a3b8",  # Muted text
                    "border": "#475569",  # Slate 600
                }

                # Signal emitter for thread-safe UI updates
                class UpdateSignal:
                    def __init__(self):
                        self.callbacks = {}

                    def connect(self, name, callback):
                        self.callbacks[name] = callback

                    def emit(self, name, *args):
                        if name in self.callbacks:
                            self.callbacks[name](*args)

                def __init__(self):
                    # Create main widget with tabs
                    self.form = QtWidgets.QWidget()
                    self.form.setWindowTitle("Conjure Dashboard")
                    self.form.setMinimumSize(400, 300)
                    self.form.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)

                    # Fix Windows redraw issues - ensure proper background painting
                    self.form.setAutoFillBackground(True)
                    self.form.setAttribute(QtCore.Qt.WA_StyledBackground, True)

                    # Apply modern stylesheet
                    self.apply_modern_style()

                    # Main layout
                    main_layout = QtWidgets.QVBoxLayout()
                    main_layout.setContentsMargins(0, 0, 0, 0)
                    main_layout.setSpacing(0)

                    # Header
                    header = self.create_header()
                    main_layout.addWidget(header)

                    # Content area with left sidebar navigation
                    content_widget = QtWidgets.QWidget()
                    content_layout = QtWidgets.QHBoxLayout()
                    content_layout.setSpacing(0)
                    content_layout.setContentsMargins(0, 0, 0, 0)

                    # Sidebar container with collapse button
                    self.sidebar_widget = QtWidgets.QWidget()
                    self.sidebar_collapsed = False
                    self.sidebar_expanded_width = 140
                    self.sidebar_collapsed_width = 50
                    sidebar_layout = QtWidgets.QVBoxLayout()
                    sidebar_layout.setContentsMargins(0, 0, 0, 0)
                    sidebar_layout.setSpacing(0)

                    # Collapse toggle button
                    self.collapse_btn = QtWidgets.QPushButton("◀")
                    self.collapse_btn.setFixedHeight(28)
                    self.collapse_btn.setToolTip("Collapse sidebar")
                    self.collapse_btn.clicked.connect(self.toggle_sidebar)
                    self.collapse_btn.setStyleSheet(f"""
                        QPushButton {{
                            background-color: {self.COLORS["bg_card"]};
                            border: none;
                            border-bottom: 1px solid {self.COLORS["border"]};
                            color: {self.COLORS["text_secondary"]};
                            font-size: 12px;
                            padding: 4px;
                            text-align: center;
                            min-width: 0px;
                        }}
                        QPushButton:hover {{
                            background-color: {self.COLORS["border"]};
                        }}
                    """)
                    sidebar_layout.addWidget(self.collapse_btn)

                    # Left sidebar navigation
                    self.nav_list = QtWidgets.QListWidget()
                    self.nav_list.setSpacing(2)
                    self.nav_list.setIconSize(QtCore.QSize(20, 20))
                    self.nav_list.setHorizontalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)
                    self.nav_list.setVerticalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)

                    # Navigation items with simple Unicode symbols (emojis don't render in Qt)
                    self.nav_items = [
                        ("◆", "Control", "Status and connections"),
                        ("▤", "Metrics", "Performance data"),
                        ("◈", "Library", "Engineering materials & standards"),
                        ("≡", "Logs", "System logs"),
                        ("●", "Config", "Configuration"),
                        ("▲", "Usage", "Rate limits"),
                    ]
                    for icon, name, tooltip in self.nav_items:
                        item = QtWidgets.QListWidgetItem(f"{icon}  {name}")
                        item.setToolTip(tooltip)
                        item.setData(QtCore.Qt.UserRole, name)  # Store name for collapsed mode
                        item.setData(QtCore.Qt.UserRole + 1, icon)  # Store icon
                        self.nav_list.addItem(item)

                    # Select first item by default
                    self.nav_list.setCurrentRow(0)
                    sidebar_layout.addWidget(self.nav_list)

                    self.sidebar_widget.setLayout(sidebar_layout)
                    # Use min/max width instead of fixed to allow window resizing
                    self.sidebar_widget.setMinimumWidth(self.sidebar_collapsed_width)
                    self.sidebar_widget.setMaximumWidth(self.sidebar_expanded_width)
                    self.sidebar_widget.setSizePolicy(QtWidgets.QSizePolicy.Fixed, QtWidgets.QSizePolicy.Expanding)

                    # Stacked widget for content panels
                    self.content_stack = QtWidgets.QStackedWidget()

                    # Create tab content widgets
                    self.control_tab = self.create_control_tab()
                    self.metrics_tab = self.create_metrics_tab()
                    self.library_tab = self.create_engineering_library_tab()
                    self.logs_tab = self.create_logs_tab()
                    self.config_tab = self.create_config_tab()
                    self.usage_tab = self.create_usage_tab()

                    # Add content widgets to stack
                    self.content_stack.addWidget(self.control_tab)
                    self.content_stack.addWidget(self.metrics_tab)
                    self.content_stack.addWidget(self.library_tab)
                    self.content_stack.addWidget(self.logs_tab)
                    self.content_stack.addWidget(self.config_tab)
                    self.content_stack.addWidget(self.usage_tab)

                    # Connect navigation to content switching
                    self.nav_list.currentRowChanged.connect(self.content_stack.setCurrentIndex)

                    # Add widgets with stretch factors for proper resizing
                    content_layout.addWidget(self.sidebar_widget, 0)  # No stretch - fixed width
                    content_layout.addWidget(self.content_stack, 1)  # Stretch to fill remaining space
                    self.content_stack.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)
                    content_widget.setLayout(content_layout)

                    main_layout.addWidget(content_widget, 1)  # Stretch content area
                    self.form.setLayout(main_layout)

                    # Initialize FreeCAD socket server for health checks
                    # Import ConjureServer from conjure module
                    self.socket_server = None
                    try:
                        import conjure

                        self.socket_server = conjure.ConjureServer()
                        self.socket_server.start()
                        print("INFO:Conjure_Gui:FreeCAD socket server started on localhost:9876")
                    except Exception as e:
                        print(f"WARNING:Conjure_Gui:Failed to start socket server: {e}")

                    # Timers for auto-refresh
                    self.timers = {}
                    self.setup_timers()

                    # Initial data load
                    self.refresh_all()

                    # Auto-connect to cloud if API key is configured
                    QtCore.QTimer.singleShot(500, self.auto_connect_cloud)

                # ===== STYLING METHODS =====

                def apply_modern_style(self):
                    """Apply modern dark theme stylesheet"""
                    c = self.COLORS
                    style = f"""
                        QWidget {{
                            background-color: {c["bg_dark"]};
                            color: {c["text_primary"]};
                            font-family: 'Segoe UI', 'SF Pro Display', -apple-system, sans-serif;
                            font-size: 13px;
                        }}
                        QListWidget {{
                            background-color: {c["bg_card"]};
                            border: none;
                            border-right: 1px solid {c["border"]};
                            outline: none;
                            padding: 8px 0;
                        }}
                        QListWidget::item {{
                            padding: 12px 16px;
                            border-radius: 0;
                            margin: 0;
                            color: {c["text_secondary"]};
                        }}
                        QListWidget::item:selected {{
                            background-color: {c["primary"]};
                            color: {c["text_primary"]};
                            border-left: 3px solid {c["primary_dark"]};
                        }}
                        QListWidget::item:hover:!selected {{
                            background-color: {c["border"]};
                            color: {c["text_primary"]};
                        }}
                        QStackedWidget {{
                            background-color: {c["bg_dark"]};
                        }}
                        QGroupBox {{
                            background-color: {c["bg_card"]};
                            border: 1px solid {c["border"]};
                            border-radius: 8px;
                            margin-top: 12px;
                            padding: 8px;
                            padding-top: 20px;
                            font-weight: 600;
                            color: {c["text_primary"]};
                        }}
                        QGroupBox::title {{
                            subcontrol-origin: margin;
                            subcontrol-position: top left;
                            left: 12px;
                            top: 4px;
                            padding: 0 6px;
                            color: {c["text_secondary"]};
                            font-size: 13px;
                        }}
                        QPushButton {{
                            background-color: {c["primary"]};
                            color: {c["text_primary"]};
                            border: none;
                            border-radius: 6px;
                            padding: 8px 16px;
                            font-weight: 500;
                            min-width: 80px;
                        }}
                        QPushButton:hover {{
                            background-color: {c["primary_dark"]};
                        }}
                        QPushButton:disabled {{
                            background-color: {c["border"]};
                            color: {c["text_secondary"]};
                        }}
                        QPushButton#dangerBtn {{
                            background-color: {c["danger"]};
                        }}
                        QPushButton#dangerBtn:hover {{
                            background-color: #dc2626;
                        }}
                        QPushButton#successBtn {{
                            background-color: {c["success"]};
                        }}
                        QPushButton#successBtn:hover {{
                            background-color: #059669;
                        }}
                        QLabel {{
                            color: {c["text_primary"]};
                        }}
                        QLabel#subtitle {{
                            color: {c["text_secondary"]};
                            font-size: 12px;
                        }}
                        QProgressBar {{
                            background-color: {c["bg_card"]};
                            border: none;
                            border-radius: 4px;
                            height: 8px;
                            text-align: center;
                        }}
                        QProgressBar::chunk {{
                            background-color: {c["primary"]};
                            border-radius: 4px;
                        }}
                        QTextEdit {{
                            background-color: {c["bg_card"]};
                            border: 1px solid {c["border"]};
                            border-radius: 6px;
                            color: {c["text_primary"]};
                            padding: 8px;
                        }}
                        QTableWidget {{
                            background-color: {c["bg_card"]};
                            border: 1px solid {c["border"]};
                            border-radius: 6px;
                            color: {c["text_primary"]};
                            gridline-color: {c["border"]};
                            alternate-background-color: rgba(255, 255, 255, 0.02);
                        }}
                        QTableWidget::item {{
                            padding: 10px 12px;
                            border-bottom: 1px solid {c["border"]};
                        }}
                        QTableWidget::item:selected {{
                            background-color: {c["primary"]};
                        }}
                        QHeaderView::section {{
                            background-color: {c["primary"]};
                            color: {c["text_primary"]};
                            padding: 10px 12px;
                            border: none;
                            border-right: 1px solid {c["primary_dark"]};
                            font-weight: 600;
                        }}
                        QHeaderView::section:last {{
                            border-right: none;
                        }}
                        QTreeWidget {{
                            background-color: {c["bg_card"]};
                            border: 1px solid {c["border"]};
                            border-radius: 6px;
                            color: {c["text_primary"]};
                            outline: none;
                        }}
                        QTreeWidget::item {{
                            padding: 6px 4px;
                            border-radius: 4px;
                        }}
                        QTreeWidget::item:selected {{
                            background-color: {c["primary"]};
                        }}
                        QTreeWidget::item:hover:!selected {{
                            background-color: {c["border"]};
                        }}
                        QScrollBar:vertical {{
                            background-color: {c["bg_dark"]};
                            width: 10px;
                            border-radius: 5px;
                            margin: 0;
                        }}
                        QScrollBar::handle:vertical {{
                            background-color: {c["border"]};
                            border-radius: 5px;
                            min-height: 30px;
                        }}
                        QScrollBar::handle:vertical:hover {{
                            background-color: {c["text_secondary"]};
                        }}
                        QScrollBar::add-line:vertical, QScrollBar::sub-line:vertical {{
                            height: 0px;
                        }}
                        QScrollBar:horizontal {{
                            background-color: {c["bg_dark"]};
                            height: 10px;
                            border-radius: 5px;
                            margin: 0;
                        }}
                        QScrollBar::handle:horizontal {{
                            background-color: {c["border"]};
                            border-radius: 5px;
                            min-width: 30px;
                        }}
                        QScrollBar::handle:horizontal:hover {{
                            background-color: {c["text_secondary"]};
                        }}
                        QScrollBar::add-line:horizontal, QScrollBar::sub-line:horizontal {{
                            width: 0px;
                        }}
                        QLineEdit {{
                            background-color: {c["bg_card"]};
                            border: 1px solid {c["border"]};
                            border-radius: 6px;
                            padding: 8px 12px;
                            color: {c["text_primary"]};
                        }}
                        QLineEdit:focus {{
                            border-color: {c["primary"]};
                        }}
                    """
                    self.form.setStyleSheet(style)

                def create_header(self):
                    """Create header with logo and title"""
                    header = QtWidgets.QWidget()
                    header.setMinimumHeight(50)
                    header.setMaximumHeight(70)
                    header.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed)
                    header.setStyleSheet(f"""
                        background-color: {self.COLORS["bg_card"]};
                        border-bottom: 1px solid {self.COLORS["border"]};
                    """)

                    layout = QtWidgets.QHBoxLayout()
                    layout.setContentsMargins(16, 0, 16, 0)

                    # Logo image
                    logo_label = QtWidgets.QLabel()
                    user_dir = App.getUserAppDataDir()
                    logo_path = os.path.join(user_dir, "Mod", "conjure", "assets", "conjure_logo.svg")
                    if os.path.exists(logo_path):
                        pixmap = QtGui.QPixmap(logo_path)
                        if not pixmap.isNull():
                            logo_label.setPixmap(pixmap.scaledToHeight(40, QtCore.Qt.SmoothTransformation))
                            logo_label.setStyleSheet("background: transparent;")
                            layout.addWidget(logo_label)

                    # Title
                    title = QtWidgets.QLabel("Conjure")
                    title.setStyleSheet("""
                        font-size: 18px;
                        font-weight: 700;
                        color: #f8fafc;
                    """)
                    layout.addWidget(title)

                    # Subtitle
                    subtitle = QtWidgets.QLabel("AI-Powered CAD Assistant")
                    subtitle.setObjectName("subtitle")
                    subtitle.setStyleSheet("font-size: 12px; color: #94a3b8;")
                    layout.addWidget(subtitle)

                    layout.addStretch()

                    # Version badge
                    version = QtWidgets.QLabel("v0.1.0")
                    version.setStyleSheet(f"""
                        background-color: {self.COLORS["primary"]};
                        color: white;
                        padding: 4px 8px;
                        border-radius: 4px;
                        font-size: 11px;
                    """)
                    layout.addWidget(version)

                    header.setLayout(layout)
                    return header

                # ===== TAB CREATION METHODS =====

                def make_scrollable(self, content_widget):
                    """Wrap a widget in a scroll area"""
                    scroll = QtWidgets.QScrollArea()
                    scroll.setWidgetResizable(True)
                    scroll.setFrameShape(QtWidgets.QFrame.NoFrame)
                    scroll.setHorizontalScrollBarPolicy(QtCore.Qt.ScrollBarAsNeeded)
                    scroll.setVerticalScrollBarPolicy(QtCore.Qt.ScrollBarAsNeeded)
                    scroll.setWidget(content_widget)
                    return scroll

                def create_control_tab(self):
                    """Create Control tab with connection status"""
                    content = QtWidgets.QWidget()
                    layout = QtWidgets.QVBoxLayout()
                    layout.setContentsMargins(16, 16, 16, 16)
                    layout.setSpacing(16)

                    # Helper to create styled labels
                    def make_label(text, color="#f8fafc", bold=False, size=None):
                        label = QtWidgets.QLabel(text)
                        style = f"color: {color};"
                        if bold:
                            style += " font-weight: bold;"
                        if size:
                            style += f" font-size: {size}px;"
                        label.setStyleSheet(style)
                        label.setMinimumHeight(20)
                        return label

                    # Hosted Server Connection group (auto-connects using env vars)
                    server_group = QtWidgets.QGroupBox("Conjure Cloud")
                    server_group.setFixedHeight(115)
                    server_layout = QtWidgets.QVBoxLayout()
                    server_layout.setSpacing(4)
                    server_layout.setContentsMargins(12, 8, 12, 8)

                    self.cloud_status = make_label("Status: ● Checking...", "#f59e0b", bold=True, size=13)
                    self.cloud_status.setFixedHeight(22)
                    server_layout.addWidget(self.cloud_status)

                    self.cloud_endpoint = make_label("Endpoint: --", "#94a3b8")
                    self.cloud_endpoint.setFixedHeight(20)
                    server_layout.addWidget(self.cloud_endpoint)

                    self.cloud_latency = make_label("Latency: --", "#f8fafc")
                    self.cloud_latency.setFixedHeight(20)
                    server_layout.addWidget(self.cloud_latency)

                    server_group.setLayout(server_layout)
                    layout.addWidget(server_group)

                    # Local FreeCAD Bridge group
                    bridge_group = QtWidgets.QGroupBox("FreeCAD Bridge")
                    bridge_group.setFixedHeight(115)
                    bridge_layout = QtWidgets.QVBoxLayout()
                    bridge_layout.setSpacing(4)
                    bridge_layout.setContentsMargins(12, 8, 12, 8)

                    self.bridge_status = make_label("Status: ● Checking...", "#f59e0b", bold=True, size=13)
                    self.bridge_status.setFixedHeight(22)
                    bridge_layout.addWidget(self.bridge_status)

                    self.bridge_port = make_label("Port: localhost:9876", "#94a3b8")
                    self.bridge_port.setFixedHeight(20)
                    bridge_layout.addWidget(self.bridge_port)

                    self.bridge_ops = make_label("Operations: 0", "#f8fafc")
                    self.bridge_ops.setFixedHeight(20)
                    bridge_layout.addWidget(self.bridge_ops)

                    bridge_group.setLayout(bridge_layout)
                    layout.addWidget(bridge_group)

                    # Quick Actions group
                    actions_group = QtWidgets.QGroupBox("Quick Actions")
                    actions_group.setFixedHeight(75)
                    actions_layout = QtWidgets.QHBoxLayout()
                    actions_layout.setSpacing(12)
                    actions_layout.setContentsMargins(12, 8, 12, 8)

                    test_btn = QtWidgets.QPushButton("Test Connection")
                    test_btn.setMinimumHeight(32)
                    test_btn.clicked.connect(self.test_connection)
                    actions_layout.addWidget(test_btn)

                    reload_btn = QtWidgets.QPushButton("Reload Bridge")
                    reload_btn.setMinimumHeight(32)
                    reload_btn.clicked.connect(self.reload_bridge)
                    actions_layout.addWidget(reload_btn)

                    actions_group.setLayout(actions_layout)
                    layout.addWidget(actions_group)

                    layout.addStretch()
                    content.setLayout(layout)
                    return self.make_scrollable(content)

                def create_metrics_tab(self):
                    """Create Metrics tab with operations data"""
                    content = QtWidgets.QWidget()
                    layout = QtWidgets.QVBoxLayout()
                    layout.setContentsMargins(16, 16, 16, 16)
                    layout.setSpacing(16)

                    # Summary group
                    summary_group = QtWidgets.QGroupBox("Operation Summary")
                    summary_layout = QtWidgets.QHBoxLayout()
                    summary_layout.setContentsMargins(16, 20, 16, 16)

                    self.ops_total = QtWidgets.QLabel("Total Operations: 0")
                    summary_layout.addWidget(self.ops_total)

                    self.ops_success = QtWidgets.QLabel("Success Rate: N/A")
                    summary_layout.addWidget(self.ops_success)

                    self.ops_latency = QtWidgets.QLabel("Avg Latency: N/A")
                    summary_layout.addWidget(self.ops_latency)

                    summary_group.setLayout(summary_layout)
                    layout.addWidget(summary_group)

                    # Operations table
                    table_group = QtWidgets.QGroupBox("Operations Breakdown")
                    table_layout = QtWidgets.QVBoxLayout()
                    table_layout.setContentsMargins(16, 20, 16, 16)

                    self.metrics_table = QtWidgets.QTableWidget()
                    self.metrics_table.setColumnCount(4)
                    self.metrics_table.setHorizontalHeaderLabels(
                        ["Operation", "Count", "Success %", "Avg Latency (ms)"]
                    )
                    self.metrics_table.horizontalHeader().setStretchLastSection(True)
                    table_layout.addWidget(self.metrics_table)

                    table_group.setLayout(table_layout)
                    layout.addWidget(table_group)

                    layout.addStretch()
                    content.setLayout(layout)
                    return self.make_scrollable(content)

                def create_engineering_library_tab(self):
                    """Create combined Engineering Library tab with Materials and Standards"""
                    widget = QtWidgets.QWidget()
                    layout = QtWidgets.QVBoxLayout()
                    layout.setContentsMargins(16, 16, 16, 16)
                    layout.setSpacing(12)

                    # View selector at top
                    selector_layout = QtWidgets.QHBoxLayout()
                    selector_label = QtWidgets.QLabel("View:")
                    selector_label.setStyleSheet("font-weight: bold;")
                    selector_layout.addWidget(selector_label)

                    self.library_selector = QtWidgets.QComboBox()
                    self.library_selector.addItem("Materials", "materials")
                    self.library_selector.addItem("Standards", "standards")
                    self.library_selector.setMinimumWidth(150)
                    self.library_selector.currentIndexChanged.connect(self._switch_library_view)
                    selector_layout.addWidget(self.library_selector)

                    selector_layout.addStretch()
                    layout.addLayout(selector_layout)

                    # Stacked widget for switching between views
                    self.library_stack = QtWidgets.QStackedWidget()

                    # Create Materials view
                    self.library_stack.addWidget(self._create_materials_view())

                    # Create Standards view
                    self.library_stack.addWidget(self._create_standards_view())

                    layout.addWidget(self.library_stack)
                    widget.setLayout(layout)

                    # Load materials on startup (standards loaded when selected)
                    QtCore.QTimer.singleShot(500, self.load_materials)

                    return widget

                def _switch_library_view(self, index):
                    """Switch between Materials and Standards views"""
                    self.library_stack.setCurrentIndex(index)
                    if index == 1:  # Standards view
                        # Load standards if not already loaded
                        if self.ext_tree.topLevelItemCount() == 0:
                            self.load_extensions()

                def _create_materials_view(self):
                    """Create the Materials view for the library tab"""
                    widget = QtWidgets.QWidget()
                    layout = QtWidgets.QVBoxLayout()
                    layout.setContentsMargins(0, 8, 0, 0)
                    layout.setSpacing(12)

                    # Status bar at top
                    self.mat_status = QtWidgets.QLabel("Material Library: Not loaded")
                    self.mat_status.setStyleSheet(
                        f"color: {self.COLORS['text_secondary']}; font-size: 11px; padding: 4px 0;"
                    )
                    layout.addWidget(self.mat_status)

                    # Search and filter bar
                    filter_layout = QtWidgets.QHBoxLayout()

                    self.mat_search = QtWidgets.QLineEdit()
                    self.mat_search.setPlaceholderText("Search materials...")
                    self.mat_search.textChanged.connect(self.filter_materials)
                    filter_layout.addWidget(self.mat_search)

                    self.mat_category = QtWidgets.QComboBox()
                    self.mat_category.addItem("All Categories", "")
                    self.mat_category.addItem("Metals", "metal")
                    self.mat_category.addItem("Plastics", "plastic")
                    self.mat_category.addItem("Composites", "composite")
                    self.mat_category.addItem("Ceramics", "ceramic")
                    self.mat_category.addItem("Elastomers", "elastomer")
                    self.mat_category.addItem("Wood", "wood")
                    self.mat_category.currentIndexChanged.connect(self.filter_materials)
                    filter_layout.addWidget(self.mat_category)

                    refresh_btn = QtWidgets.QPushButton("Refresh")
                    refresh_btn.clicked.connect(self.load_materials)
                    filter_layout.addWidget(refresh_btn)

                    layout.addLayout(filter_layout)

                    # Split view: list on left, details on right
                    splitter = QtWidgets.QSplitter(QtCore.Qt.Horizontal)

                    # Materials list
                    self.mat_list = QtWidgets.QListWidget()
                    self.mat_list.setMinimumWidth(200)
                    self.mat_list.itemClicked.connect(self.show_material_details)
                    splitter.addWidget(self.mat_list)

                    # Details panel
                    details_widget = QtWidgets.QWidget()
                    details_layout = QtWidgets.QVBoxLayout()
                    details_layout.setContentsMargins(12, 0, 0, 0)
                    details_layout.setSpacing(8)

                    self.mat_title = QtWidgets.QLabel("Select a material")
                    self.mat_title.setStyleSheet("font-size: 16px; font-weight: 600; padding: 4px 0;")
                    details_layout.addWidget(self.mat_title)

                    self.mat_details = QtWidgets.QTextEdit()
                    self.mat_details.setReadOnly(True)
                    self.mat_details.setStyleSheet("font-family: 'Consolas', 'Monaco', monospace; font-size: 12px;")
                    details_layout.addWidget(self.mat_details)

                    # Assign material button
                    self.mat_assign_btn = QtWidgets.QPushButton("Assign to Selected Object")
                    self.mat_assign_btn.setObjectName("successBtn")
                    self.mat_assign_btn.clicked.connect(self.assign_material_to_selected)
                    self.mat_assign_btn.setEnabled(False)
                    details_layout.addWidget(self.mat_assign_btn)

                    details_widget.setLayout(details_layout)
                    splitter.addWidget(details_widget)

                    splitter.setSizes([200, 350])
                    layout.addWidget(splitter)

                    # Selected object info
                    obj_group = QtWidgets.QGroupBox("Selected Object")
                    obj_layout = QtWidgets.QVBoxLayout()
                    obj_layout.setContentsMargins(12, 20, 12, 12)

                    self.mat_obj_label = QtWidgets.QLabel("No object selected")
                    obj_layout.addWidget(self.mat_obj_label)

                    self.mat_obj_material = QtWidgets.QLabel("Material: None")
                    obj_layout.addWidget(self.mat_obj_material)

                    clear_btn = QtWidgets.QPushButton("Clear Material")
                    clear_btn.setObjectName("dangerBtn")
                    clear_btn.clicked.connect(self.clear_selected_material)
                    obj_layout.addWidget(clear_btn)

                    obj_group.setLayout(obj_layout)
                    layout.addWidget(obj_group)

                    widget.setLayout(layout)

                    # Store materials data
                    self._materials_data = []
                    self._selected_material_id = None

                    return widget

                def _create_standards_view(self):
                    """Create the Standards view for the library tab"""
                    widget = QtWidgets.QWidget()
                    layout = QtWidgets.QVBoxLayout()
                    layout.setContentsMargins(0, 8, 0, 0)
                    layout.setSpacing(12)

                    # Search bar
                    search_layout = QtWidgets.QHBoxLayout()
                    self.ext_search = QtWidgets.QLineEdit()
                    self.ext_search.setPlaceholderText("Search standards...")
                    self.ext_search.textChanged.connect(self.filter_extensions)
                    search_layout.addWidget(self.ext_search)

                    refresh_btn = QtWidgets.QPushButton("Refresh")
                    refresh_btn.clicked.connect(self.load_extensions)
                    search_layout.addWidget(refresh_btn)
                    layout.addLayout(search_layout)

                    # Split view: tree on left, details on right
                    splitter = QtWidgets.QSplitter(QtCore.Qt.Horizontal)

                    # Category tree
                    self.ext_tree = QtWidgets.QTreeWidget()
                    self.ext_tree.setHeaderLabels(["Category / Item"])
                    self.ext_tree.setMinimumWidth(200)
                    self.ext_tree.itemClicked.connect(self.show_extension_details)
                    splitter.addWidget(self.ext_tree)

                    # Details panel
                    details_widget = QtWidgets.QWidget()
                    details_layout = QtWidgets.QVBoxLayout()
                    details_layout.setContentsMargins(12, 0, 0, 0)
                    details_layout.setSpacing(8)

                    self.ext_title = QtWidgets.QLabel("Select an item")
                    self.ext_title.setStyleSheet("font-size: 16px; font-weight: 600; padding: 4px 0;")
                    details_layout.addWidget(self.ext_title)

                    self.ext_details = QtWidgets.QTextEdit()
                    self.ext_details.setReadOnly(True)
                    self.ext_details.setStyleSheet("font-family: 'Consolas', 'Monaco', monospace; font-size: 12px;")
                    details_layout.addWidget(self.ext_details)

                    details_widget.setLayout(details_layout)
                    splitter.addWidget(details_widget)

                    splitter.setSizes([250, 350])
                    layout.addWidget(splitter)

                    # Status bar
                    self.ext_status = QtWidgets.QLabel("Select 'Standards' to load")
                    self.ext_status.setStyleSheet(
                        f"color: {self.COLORS['text_secondary']}; font-size: 11px; padding: 4px 0;"
                    )
                    layout.addWidget(self.ext_status)

                    widget.setLayout(layout)
                    return widget

                def load_extensions(self):
                    """Load extensions data from server API"""
                    self.ext_tree.clear()
                    self._extensions_data = {}
                    self._pending_extensions = None
                    self._extensions_error = None

                    def fetch_in_thread():
                        """Fetch extensions from API in background thread"""
                        try:
                            from urllib.error import URLError
                            from urllib.request import Request, urlopen

                            # Try server endpoints (public catalog endpoint)
                            endpoints = [
                                "http://localhost:8080/api/v1/standards",
                                "https://conjure.lautrek.com/api/v1/standards",
                            ]

                            for endpoint in endpoints:
                                try:
                                    req = Request(endpoint)
                                    req.add_header("User-Agent", "Conjure-FreeCAD/0.1.0")
                                    with urlopen(req, timeout=5) as response:
                                        if response.status == 200:
                                            self._pending_extensions = json.loads(response.read().decode("utf-8"))
                                            return
                                except (URLError, Exception):
                                    continue

                            self._extensions_error = "Server unavailable - start with 'make up'"

                        except Exception as e:
                            self._extensions_error = f"Error: {e}"

                    def check_result():
                        """Check if background fetch completed (called from main thread timer)"""
                        if self._pending_extensions is not None:
                            self._populate_extensions(self._pending_extensions)
                            self._pending_extensions = None
                        elif self._extensions_error is not None:
                            self.ext_status.setText(self._extensions_error)
                            self._extensions_error = None
                        else:
                            # Still loading, check again
                            QtCore.QTimer.singleShot(100, check_result)

                    # Run fetch in background thread
                    self.ext_status.setText("Loading extensions from server...")
                    fetch_thread = threading.Thread(target=fetch_in_thread, daemon=True)
                    fetch_thread.start()

                    # Start polling timer on main thread
                    QtCore.QTimer.singleShot(100, check_result)

                def _populate_extensions(self, response_data):
                    """Populate tree with extensions data (called from main thread)"""
                    try:
                        standards = response_data.get("standards", {})
                        total_items = 0

                        categories = {
                            "sockets": ("Sockets", "Utility key sockets"),
                            "fasteners": ("Fasteners", "Bolts, screws, nuts"),
                            "materials": ("Materials", "3D printing materials"),
                            "threads": ("Threads", "Thread specifications"),
                            "profiles": ("Profiles", "Aluminum extrusions"),
                            "hardware": ("Hardware", "General hardware"),
                        }

                        for cat_key, (cat_name, cat_desc) in categories.items():
                            if cat_key not in standards:
                                continue

                            items = standards[cat_key]
                            if not items:
                                continue

                            # Create category item
                            cat_item = QtWidgets.QTreeWidgetItem([f"{cat_name} ({len(items)})"])
                            cat_item.setData(
                                0, QtCore.Qt.UserRole, {"type": "category", "key": cat_key, "description": cat_desc}
                            )
                            self.ext_tree.addTopLevelItem(cat_item)

                            # Add items
                            for item_id in sorted(items):
                                child = QtWidgets.QTreeWidgetItem([item_id])
                                child.setData(
                                    0, QtCore.Qt.UserRole, {"type": "item", "category": cat_key, "id": item_id}
                                )
                                cat_item.addChild(child)
                                total_items += 1

                        self.ext_status.setText(f"Loaded {total_items} items from server")

                    except Exception as e:
                        self.ext_status.setText(f"Error parsing extensions: {e}")

                def get_standard_from_api(self, category, spec_id):
                    """Fetch a specific standard from the server API"""
                    try:
                        from urllib.request import Request, urlopen

                        # Use public endpoint (no auth required)
                        endpoints = [
                            f"http://localhost:8080/api/v1/standards/{category}/{spec_id}",
                            f"https://conjure.lautrek.com/api/v1/standards/{category}/{spec_id}",
                        ]

                        for endpoint in endpoints:
                            try:
                                req = Request(endpoint)
                                req.add_header("User-Agent", "Conjure-FreeCAD/0.1.0")
                                with urlopen(req, timeout=5) as response:
                                    if response.status == 200:
                                        data = json.loads(response.read().decode("utf-8"))
                                        return data.get("spec", data)
                            except Exception:
                                continue
                        return None
                    except Exception:
                        return None

                def filter_extensions(self, text):
                    """Filter tree items by search text"""
                    text = text.lower()
                    for i in range(self.ext_tree.topLevelItemCount()):
                        cat_item = self.ext_tree.topLevelItem(i)
                        cat_visible = False
                        for j in range(cat_item.childCount()):
                            child = cat_item.child(j)
                            visible = text in child.text(0).lower() or not text
                            child.setHidden(not visible)
                            if visible:
                                cat_visible = True
                        cat_item.setHidden(not cat_visible and bool(text))
                        if cat_visible and text:
                            cat_item.setExpanded(True)

                def show_extension_details(self, item, column):
                    """Show details for selected extension item"""
                    data = item.data(0, QtCore.Qt.UserRole)
                    if not data:
                        return

                    if data["type"] == "category":
                        self.ext_title.setText(data["key"].title())
                        self.ext_details.setPlainText(
                            f"Category: {data['key']}\n\n{data['description']}\n\nSelect an item to view details."
                        )
                        return

                    if data["type"] == "item":
                        spec = self.get_standard_from_api(data["category"], data["id"])
                        if spec:
                            self.ext_title.setText(data["id"])
                            # Format spec as readable text
                            lines = [f"ID: {data['id']}", f"Category: {data['category']}", ""]
                            for key, value in spec.items():
                                if key == "id":
                                    continue
                                if isinstance(value, dict):
                                    lines.append(f"{key}:")
                                    for k, v in value.items():
                                        lines.append(f"  {k}: {v}")
                                elif isinstance(value, list):
                                    lines.append(f"{key}: {', '.join(str(v) for v in value)}")
                                else:
                                    lines.append(f"{key}: {value}")
                            self.ext_details.setPlainText("\n".join(lines))
                        else:
                            self.ext_details.setPlainText("Could not load details.")

                def load_materials(self):
                    """Load materials from server via socket command"""
                    self.mat_list.clear()
                    self._materials_data = []
                    self.mat_status.setText("Loading materials...")

                    def fetch_in_thread():
                        """Fetch materials in background thread"""
                        try:
                            result = self.send_socket_command({"type": "list_engineering_materials"})
                            if result.get("status") == "success":
                                self._pending_materials = result.get("materials", [])
                            else:
                                self._materials_error = result.get("error", "Unknown error")
                        except Exception as e:
                            self._materials_error = str(e)

                    def check_result():
                        """Check if background fetch completed"""
                        if hasattr(self, "_pending_materials") and self._pending_materials is not None:
                            self._populate_materials(self._pending_materials)
                            self._pending_materials = None
                        elif hasattr(self, "_materials_error") and self._materials_error is not None:
                            self.mat_status.setText(f"Error: {self._materials_error}")
                            self._materials_error = None
                        else:
                            QtCore.QTimer.singleShot(100, check_result)

                    self._pending_materials = None
                    self._materials_error = None

                    fetch_thread = threading.Thread(target=fetch_in_thread, daemon=True)
                    fetch_thread.start()
                    QtCore.QTimer.singleShot(100, check_result)

                def _populate_materials(self, materials):
                    """Populate materials list from data"""
                    self._materials_data = materials
                    self.mat_list.clear()

                    for mat in materials:
                        item = QtWidgets.QListWidgetItem(f"{mat.get('name', 'Unknown')}")
                        item.setData(QtCore.Qt.UserRole, mat)
                        # Color-code by category
                        category = mat.get("category", "")
                        if category == "metal":
                            item.setForeground(QtGui.QColor("#60a5fa"))  # Blue
                        elif category == "plastic":
                            item.setForeground(QtGui.QColor("#34d399"))  # Green
                        elif category == "composite":
                            item.setForeground(QtGui.QColor("#a78bfa"))  # Purple
                        elif category == "ceramic":
                            item.setForeground(QtGui.QColor("#fbbf24"))  # Yellow
                        self.mat_list.addItem(item)

                    self.mat_status.setText(f"Loaded {len(materials)} materials")

                def filter_materials(self):
                    """Filter materials list by search text and category"""
                    search_text = self.mat_search.text().lower()
                    category = self.mat_category.currentData()

                    for i in range(self.mat_list.count()):
                        item = self.mat_list.item(i)
                        mat = item.data(QtCore.Qt.UserRole)

                        matches_search = not search_text or search_text in mat.get("name", "").lower()
                        matches_category = not category or mat.get("category") == category

                        item.setHidden(not (matches_search and matches_category))

                def show_material_details(self, item):
                    """Show details for selected material"""
                    mat = item.data(QtCore.Qt.UserRole)
                    if not mat:
                        return

                    self._selected_material_id = mat.get("id")
                    self.mat_title.setText(mat.get("name", "Unknown"))
                    self.mat_assign_btn.setEnabled(True)

                    # Format details
                    lines = [
                        f"ID: {mat.get('id', 'N/A')}",
                        f"Category: {mat.get('category', 'N/A').capitalize()}",
                        "",
                        "Mechanical Properties:",
                    ]

                    mech = mat.get("mechanical", {})
                    if mech.get("density_kg_m3"):
                        lines.append(f"  Density: {mech['density_kg_m3']:.0f} kg/m³")
                    if mech.get("youngs_modulus_pa"):
                        e_gpa = mech["youngs_modulus_pa"] / 1e9
                        lines.append(f"  Young's Modulus: {e_gpa:.1f} GPa")
                    if mech.get("poissons_ratio"):
                        lines.append(f"  Poisson's Ratio: {mech['poissons_ratio']:.2f}")
                    if mech.get("yield_strength_pa"):
                        ys_mpa = mech["yield_strength_pa"] / 1e6
                        lines.append(f"  Yield Strength: {ys_mpa:.0f} MPa")
                    if mech.get("ultimate_strength_pa"):
                        us_mpa = mech["ultimate_strength_pa"] / 1e6
                        lines.append(f"  Ultimate Strength: {us_mpa:.0f} MPa")

                    lines.append("")
                    lines.append("Thermal Properties:")

                    thermal = mat.get("thermal", {})
                    if thermal.get("thermal_conductivity_w_mk"):
                        lines.append(f"  Thermal Conductivity: {thermal['thermal_conductivity_w_mk']:.1f} W/m·K")
                    if thermal.get("specific_heat_j_kgk"):
                        lines.append(f"  Specific Heat: {thermal['specific_heat_j_kgk']:.0f} J/kg·K")
                    if thermal.get("melting_point_c"):
                        lines.append(f"  Melting Point: {thermal['melting_point_c']:.0f} °C")

                    if mat.get("description"):
                        lines.append("")
                        lines.append(f"Description: {mat['description']}")

                    self.mat_details.setPlainText("\n".join(lines))

                def assign_material_to_selected(self):
                    """Assign the selected material to the selected FreeCAD object"""
                    if not self._selected_material_id:
                        self.log("No material selected")
                        return

                    # Get selected object from FreeCAD
                    selection = Gui.Selection.getSelection()
                    if not selection:
                        self.log("No object selected in FreeCAD")
                        return

                    obj = selection[0]
                    obj_name = obj.Name

                    # Send assign command
                    result = self.send_socket_command(
                        {
                            "type": "assign_engineering_material",
                            "params": {"object_name": obj_name, "material_id": self._selected_material_id},
                        }
                    )

                    if result.get("status") == "success":
                        material_name = result.get("material_name", self._selected_material_id)
                        self.log(f"Assigned {material_name} to {obj_name}")
                        self.mat_obj_label.setText(f"Object: {obj_name}")
                        self.mat_obj_material.setText(f"Material: {material_name}")
                    else:
                        self.log(f"Failed to assign material: {result.get('error', 'Unknown error')}")

                def clear_selected_material(self):
                    """Clear material from selected object"""
                    selection = Gui.Selection.getSelection()
                    if not selection:
                        self.log("No object selected in FreeCAD")
                        return

                    obj = selection[0]
                    obj_name = obj.Name

                    result = self.send_socket_command(
                        {"type": "clear_engineering_material", "params": {"object_name": obj_name}}
                    )

                    if result.get("status") == "success":
                        self.log(f"Cleared material from {obj_name}")
                        self.mat_obj_label.setText(f"Object: {obj_name}")
                        self.mat_obj_material.setText("Material: None")
                    else:
                        self.log(f"Failed to clear material: {result.get('error', 'Unknown error')}")

                def create_logs_tab(self):
                    """Create Logs tab with live log viewer"""
                    widget = QtWidgets.QWidget()
                    layout = QtWidgets.QVBoxLayout()
                    layout.setContentsMargins(16, 16, 16, 16)
                    layout.setSpacing(12)

                    # Log viewer
                    self.logs_display = QtWidgets.QTextEdit()
                    self.logs_display.setReadOnly(True)
                    self.logs_display.setFont(QtGui.QFont("Courier", 9))
                    layout.addWidget(self.logs_display)

                    # Control buttons
                    btn_layout = QtWidgets.QHBoxLayout()

                    refresh_btn = QtWidgets.QPushButton("Refresh Now")
                    refresh_btn.clicked.connect(self.refresh_logs)
                    btn_layout.addWidget(refresh_btn)

                    clear_btn = QtWidgets.QPushButton("Clear Display")
                    clear_btn.clicked.connect(self.logs_display.clear)
                    btn_layout.addWidget(clear_btn)

                    btn_layout.addStretch()
                    layout.addLayout(btn_layout)

                    widget.setLayout(layout)
                    return widget

                def create_config_tab(self):
                    """Create Config tab with server configuration"""
                    widget = QtWidgets.QWidget()
                    layout = QtWidgets.QVBoxLayout()
                    layout.setContentsMargins(16, 16, 16, 16)
                    layout.setSpacing(12)

                    # Config display
                    self.config_display = QtWidgets.QTextEdit()
                    self.config_display.setReadOnly(True)
                    self.config_display.setFont(QtGui.QFont("Courier", 9))
                    layout.addWidget(self.config_display)

                    # Refresh button
                    btn_layout = QtWidgets.QHBoxLayout()
                    refresh_btn = QtWidgets.QPushButton("Refresh Config")
                    refresh_btn.clicked.connect(self.refresh_config)
                    btn_layout.addWidget(refresh_btn)
                    btn_layout.addStretch()
                    layout.addLayout(btn_layout)

                    widget.setLayout(layout)
                    return widget

                def create_usage_tab(self):
                    """Create Usage tab with rate limit tracking"""
                    content = QtWidgets.QWidget()
                    layout = QtWidgets.QVBoxLayout()
                    layout.setContentsMargins(16, 16, 16, 16)
                    layout.setSpacing(16)

                    # Tier info
                    tier_group = QtWidgets.QGroupBox("Current Tier")
                    tier_layout = QtWidgets.QVBoxLayout()
                    tier_layout.setContentsMargins(16, 28, 16, 16)

                    self.tier_label = QtWidgets.QLabel("Tier: Unknown")
                    self.tier_label.setStyleSheet("font-weight: bold; font-size: 12px;")
                    tier_layout.addWidget(self.tier_label)

                    tier_group.setLayout(tier_layout)
                    layout.addWidget(tier_group)

                    # Usage bars
                    usage_group = QtWidgets.QGroupBox("Rate Limit Usage")
                    usage_layout = QtWidgets.QVBoxLayout()
                    usage_layout.setContentsMargins(16, 28, 16, 16)
                    usage_layout.setSpacing(12)

                    self.usage_bar = QtWidgets.QProgressBar()
                    self.usage_bar.setRange(0, 100)
                    usage_layout.addWidget(QtWidgets.QLabel("Operations"))
                    usage_layout.addWidget(self.usage_bar)

                    self.usage_label = QtWidgets.QLabel("Used: 0 / 5000")
                    usage_layout.addWidget(self.usage_label)

                    usage_group.setLayout(usage_layout)
                    layout.addWidget(usage_group)

                    layout.addStretch()
                    content.setLayout(layout)
                    return self.make_scrollable(content)

                # ===== DATA COLLECTION METHODS =====

                def send_socket_command(self, command_dict, timeout=3.0):
                    """Send command to server via socket"""
                    try:
                        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
                        sock.settimeout(timeout)
                        sock.connect(("localhost", 9876))
                        sock.sendall(json.dumps(command_dict).encode("utf-8") + b"\n")
                        response = sock.recv(8192).decode("utf-8")
                        sock.close()
                        return json.loads(response)
                    except socket.timeout:
                        return {"status": "error", "message": "Socket timeout"}
                    except Exception as e:
                        return {"status": "error", "message": str(e)}

                def get_real_health_status(self):
                    """Get real health status via socket"""
                    return self.send_socket_command({"type": "health_check"})

                def get_metrics_data(self):
                    """Get metrics data from server"""
                    return self.send_socket_command({"type": "get_metrics"})

                def get_recent_logs(self, num_lines=50):
                    """Get recent log entries from file"""
                    try:
                        import os

                        log_file = "/tmp/conjure_server.log"
                        if not os.path.exists(log_file):
                            return []
                        with open(log_file) as f:
                            lines = f.readlines()
                        return lines[-num_lines:]
                    except Exception as e:
                        return [f"Error reading logs: {str(e)}"]

                def get_config_data(self):
                    """Get configuration data from server"""
                    return self.send_socket_command({"type": "get_config"})

                def get_usage_stats(self):
                    """Get usage statistics from server"""
                    return self.send_socket_command({"type": "get_usage"})

                # ===== UI UPDATE METHODS =====

                def update_health_ui(self):
                    """Update connection status displays"""
                    import urllib.request

                    from conjure import get_cloud_bridge

                    # Check cloud bridge (WebSocket connection)
                    bridge = get_cloud_bridge()
                    if bridge and bridge.running and bridge.adapter_id:
                        # Cloud bridge is connected via WebSocket
                        self.cloud_status.setText("Status: ● Bridge Connected")
                        self.cloud_status.setStyleSheet(
                            f"font-weight: bold; font-size: 14px; color: {self.COLORS['success']};"
                        )
                        # Extract hostname from bridge URL
                        from urllib.parse import urlparse

                        parsed = urlparse(bridge.server_url)
                        endpoint = parsed.hostname or "unknown"
                        self.cloud_endpoint.setText(f"Endpoint: {endpoint} (WS)")
                        self.cloud_latency.setText(f"Adapter: {bridge.adapter_id}")
                    elif bridge and bridge.running:
                        # Bridge is connecting
                        self.cloud_status.setText("Status: ● Connecting...")
                        self.cloud_status.setStyleSheet(
                            f"font-weight: bold; font-size: 14px; color: {self.COLORS['warning']};"
                        )
                    else:
                        # Check cloud server reachability (HTTP health check)
                        cloud_ok = False
                        latency_ms = 0
                        try:
                            start = time.time()
                            endpoints = [
                                ("http://localhost:8080/health", "localhost:8080"),
                                ("https://conjure.lautrek.com/health", "conjure.lautrek.com"),
                            ]
                            for url, endpoint in endpoints:
                                try:
                                    req = urllib.request.Request(url, method="GET")
                                    req.add_header("User-Agent", "Conjure-FreeCAD/0.1.0")
                                    with urllib.request.urlopen(req, timeout=2) as resp:
                                        if resp.status == 200:
                                            cloud_ok = True
                                            latency_ms = int((time.time() - start) * 1000)
                                            self.cloud_endpoint.setText(f"Endpoint: {endpoint}")
                                            break
                                except:
                                    continue
                        except:
                            pass

                        if cloud_ok:
                            self.cloud_status.setText("Status: ● Reachable")
                            self.cloud_status.setStyleSheet(
                                f"font-weight: bold; font-size: 14px; color: {self.COLORS['warning']};"
                            )
                            self.cloud_latency.setText(f"Latency: {latency_ms}ms")
                        else:
                            self.cloud_status.setText("Status: ● Unreachable")
                            self.cloud_status.setStyleSheet(
                                f"font-weight: bold; font-size: 14px; color: {self.COLORS['danger']};"
                            )
                            self.cloud_latency.setText("Latency: --")

                    # Check local bridge (socket server)
                    bridge_health = self.get_real_health_status()
                    if bridge_health.get("status") == "success":
                        self.bridge_status.setText("Status: ● Running")
                        self.bridge_status.setStyleSheet(
                            f"font-weight: bold; font-size: 14px; color: {self.COLORS['success']};"
                        )
                        ops_count = bridge_health.get("operations_count", 0)
                        self.bridge_ops.setText(f"Operations: {ops_count}")
                    else:
                        self.bridge_status.setText("Status: ● Not Running")
                        self.bridge_status.setStyleSheet(
                            f"font-weight: bold; font-size: 14px; color: {self.COLORS['danger']};"
                        )
                        self.bridge_ops.setText("Operations: --")

                def update_metrics_ui(self):
                    """Update metrics display"""
                    metrics = self.get_metrics_data()
                    if metrics.get("status") != "success":
                        self.ops_total.setText("Total Operations: N/A")
                        self.ops_success.setText("Success Rate: N/A")
                        self.ops_latency.setText("Avg Latency: N/A")
                        self.metrics_table.setRowCount(0)
                        return

                    metrics_data = metrics.get("metrics", {})
                    # Placeholder implementation
                    self.ops_total.setText(f"Total Operations: {len(metrics_data)}")
                    self.ops_success.setText("Success Rate: Data pending")
                    self.ops_latency.setText("Avg Latency: Data pending")

                def update_logs_ui(self):
                    """Update logs display"""
                    logs = self.get_recent_logs(50)
                    html_content = "<pre style='background-color: #f5f5f5; padding: 10px;'>"
                    for line in logs:
                        line = line.rstrip()
                        if "ERROR" in line:
                            html_content += f"<span style='color: red;'>{self.escape_html(line)}</span>\n"
                        elif "WARNING" in line:
                            html_content += f"<span style='color: darkorange;'>{self.escape_html(line)}</span>\n"
                        else:
                            html_content += f"<span style='color: black;'>{self.escape_html(line)}</span>\n"
                    html_content += "</pre>"
                    self.logs_display.setHtml(html_content)

                def update_config_ui(self):
                    """Update config display"""
                    config = self.get_config_data()
                    if config.get("status") != "success":
                        self.config_display.setText("Configuration not available")
                        return

                    config_data = config.get("config", {})
                    config_text = "Conjure Server Configuration\n" + "=" * 50 + "\n\n"
                    for section, values in config_data.items():
                        config_text += f"[{section.upper()}]\n"
                        for key, value in values.items():
                            config_text += f"  {key}: {value}\n"
                        config_text += "\n"

                    config_file = config.get("config_file", "Not specified")
                    config_text += f"\nConfiguration file: {config_file}"
                    self.config_display.setText(config_text)

                def update_usage_ui(self):
                    """Update usage display"""
                    usage = self.get_usage_stats()
                    if usage.get("status") != "success":
                        self.tier_label.setText("Tier: Unavailable")
                        self.usage_bar.setValue(0)
                        self.usage_label.setText("Usage data unavailable")
                        return

                    tier = usage.get("tier", "unknown").capitalize()
                    self.tier_label.setText(f"Tier: {tier}")

                    usage_data = usage.get("usage", {})
                    used = usage_data.get("operations_recent", 0)
                    limit = usage_data.get("limit", 5000)
                    percent = int(used / limit * 100) if limit > 0 else 0

                    self.usage_bar.setValue(percent)
                    self.usage_label.setText(f"Used: {used} / {limit}")

                # ===== TIMER CALLBACKS =====

                def refresh_health(self):
                    """Refresh health status (synchronous, safe for UI)"""
                    try:
                        self.update_health_ui()
                    except Exception as e:
                        print(f"ERROR:Conjure_Gui:Error refreshing health: {e}")

                def refresh_metrics(self):
                    """Refresh metrics (synchronous, safe for UI)"""
                    try:
                        self.update_metrics_ui()
                    except Exception as e:
                        print(f"ERROR:Conjure_Gui:Error refreshing metrics: {e}")

                def refresh_logs(self):
                    """Refresh logs (synchronous, safe for UI)"""
                    try:
                        self.update_logs_ui()
                    except Exception as e:
                        print(f"ERROR:Conjure_Gui:Error refreshing logs: {e}")

                def refresh_config(self):
                    """Refresh config (synchronous, safe for UI)"""
                    try:
                        self.update_config_ui()
                    except Exception as e:
                        print(f"ERROR:Conjure_Gui:Error refreshing config: {e}")

                def refresh_usage(self):
                    """Refresh usage (synchronous, safe for UI)"""
                    try:
                        self.update_usage_ui()
                    except Exception as e:
                        print(f"ERROR:Conjure_Gui:Error refreshing usage: {e}")

                def load_api_key(self):
                    """Load API key from environment or config file."""
                    import os

                    # Check environment first
                    api_key = os.environ.get("CONJURE_API_KEY", "")
                    if api_key:
                        return api_key

                    # Check config file
                    try:
                        import yaml

                        config_paths = [
                            os.path.expanduser("~/.conjure/config.yaml"),
                            os.path.expanduser("~/.config/conjure/config.yaml"),
                        ]
                        for path in config_paths:
                            if os.path.exists(path):
                                with open(path) as f:
                                    config = yaml.safe_load(f)
                                    if config and config.get("hosted_server", {}).get("api_key"):
                                        return config["hosted_server"]["api_key"]
                    except Exception:
                        pass

                    return ""

                def load_server_url(self):
                    """Load server URL from env var, config file, or default."""
                    import os

                    # 1. Check environment variable first
                    env_url = os.environ.get("CONJURE_SERVER_URL", "")
                    if env_url:
                        # Convert HTTP URL to WebSocket URL
                        ws_url = env_url.replace("https://", "wss://").replace("http://", "ws://")
                        # Add WebSocket path if not present
                        if "/api/v1/adapter/ws" not in ws_url:
                            ws_url = ws_url.rstrip("/") + "/api/v1/adapter/ws"
                        return ws_url

                    # 2. Check config file
                    try:
                        import yaml

                        config_path = os.path.expanduser("~/.conjure/config.yaml")
                        if os.path.exists(config_path):
                            with open(config_path) as f:
                                config = yaml.safe_load(f)
                                if config and config.get("hosted_server", {}).get("ws_url"):
                                    return config["hosted_server"]["ws_url"]
                    except Exception:
                        pass

                    # 3. Default to production
                    return "wss://conjure.lautrek.com/api/v1/adapter/ws"

                def save_api_key(self, api_key, server_url=None):
                    """Save API key to config file."""
                    import os

                    config_dir = os.path.expanduser("~/.conjure")
                    config_path = os.path.join(config_dir, "config.yaml")

                    try:
                        import yaml

                        os.makedirs(config_dir, exist_ok=True)

                        # Load existing config or create new
                        config = {}
                        if os.path.exists(config_path):
                            with open(config_path) as f:
                                config = yaml.safe_load(f) or {}

                        # Update API key and server URL
                        if "hosted_server" not in config:
                            config["hosted_server"] = {}
                        config["hosted_server"]["api_key"] = api_key
                        if server_url:
                            config["hosted_server"]["ws_url"] = server_url
                            # Derive HTTP URL from WS URL
                            http_url = server_url.replace("wss://", "https://").replace("ws://", "http://")
                            http_url = http_url.split("/api/")[0]  # Strip path
                            config["hosted_server"]["url"] = http_url
                        config["hosted_server"]["enabled"] = True

                        # Save
                        with open(config_path, "w") as f:
                            yaml.dump(config, f, default_flow_style=False)

                        self.log(f"API key saved to {config_path}")
                    except Exception as e:
                        self.log(f"Failed to save API key: {e}")

                def toggle_cloud_bridge(self, force_connect=False):
                    """Connect or disconnect the cloud bridge."""
                    from conjure import get_cloud_bridge, start_cloud_bridge, stop_cloud_bridge

                    bridge = get_cloud_bridge()

                    if bridge and bridge.running and not force_connect:
                        # Disconnect
                        stop_cloud_bridge()
                        self.cloud_status.setText("Status: ● Disconnected")
                        self.cloud_status.setStyleSheet(
                            f"font-weight: bold; font-size: 14px; color: {self.COLORS['danger']};"
                        )
                        self.cloud_endpoint.setText("Endpoint: --")
                        self.log("Cloud bridge disconnected")
                    else:
                        # Connect using env vars / config
                        api_key = self.load_api_key()
                        if not api_key:
                            self.cloud_status.setText("Status: ● No API Key")
                            self.cloud_status.setStyleSheet(
                                f"font-weight: bold; font-size: 14px; color: {self.COLORS['danger']};"
                            )
                            self.log("No API key configured. Set CONJURE_API_KEY env var or ~/.conjure/config.yaml")
                            return

                        server_url = self.load_server_url()

                        # Extract host for display
                        try:
                            from urllib.parse import urlparse

                            parsed = urlparse(server_url.replace("wss://", "https://").replace("ws://", "http://"))
                            display_host = parsed.netloc
                        except:
                            display_host = server_url

                        try:
                            start_cloud_bridge(api_key, server_url)
                            self.cloud_status.setText("Status: ● Connecting...")
                            self.cloud_status.setStyleSheet(
                                f"font-weight: bold; font-size: 14px; color: {self.COLORS['warning']};"
                            )
                            self.cloud_endpoint.setText(f"Endpoint: {display_host}")
                            self.log(f"Cloud bridge connecting to {display_host}...")
                        except Exception as e:
                            self.log(f"Error starting cloud bridge: {e}")

                def auto_connect_cloud(self):
                    """Auto-connect to cloud if API key is configured."""
                    api_key = self.load_api_key()
                    if api_key:
                        self.log("Auto-connecting to Conjure Cloud...")
                        self.toggle_cloud_bridge(force_connect=True)
                    else:
                        self.cloud_status.setText("Status: ● Not Configured")
                        self.cloud_status.setStyleSheet(
                            f"font-weight: bold; font-size: 14px; color: {self.COLORS['text_muted']};"
                        )
                        self.log("No API key found. Set CONJURE_API_KEY env var to auto-connect.")

                def refresh_all(self):
                    """Refresh all data"""
                    self.refresh_health()
                    # Skip other refreshes on initial load to avoid delays
                    # They'll refresh on their own timers

                def toggle_sidebar(self):
                    """Toggle sidebar between collapsed (icons only) and expanded (icons + text)"""
                    self.sidebar_collapsed = not self.sidebar_collapsed

                    if self.sidebar_collapsed:
                        # Collapse: show only icons, centered
                        self.sidebar_widget.setMinimumWidth(self.sidebar_collapsed_width)
                        self.sidebar_widget.setMaximumWidth(self.sidebar_collapsed_width)
                        self.collapse_btn.setText("▶")
                        self.collapse_btn.setToolTip("Expand sidebar")
                        # Update items to show only icons (centered)
                        for i in range(self.nav_list.count()):
                            item = self.nav_list.item(i)
                            icon = item.data(QtCore.Qt.UserRole + 1)
                            item.setText(icon)
                            item.setTextAlignment(QtCore.Qt.AlignCenter)
                    else:
                        # Expand: show icons + text, left aligned
                        self.sidebar_widget.setMinimumWidth(self.sidebar_collapsed_width)
                        self.sidebar_widget.setMaximumWidth(self.sidebar_expanded_width)
                        self.collapse_btn.setText("◀")
                        self.collapse_btn.setToolTip("Collapse sidebar")
                        # Update items to show icons + text (left aligned)
                        for i in range(self.nav_list.count()):
                            item = self.nav_list.item(i)
                            icon = item.data(QtCore.Qt.UserRole + 1)
                            name = item.data(QtCore.Qt.UserRole)
                            item.setText(f"{icon}  {name}")
                            item.setTextAlignment(QtCore.Qt.AlignLeft | QtCore.Qt.AlignVCenter)

                def setup_timers(self):
                    """Setup auto-refresh timers"""
                    # Health timer (5 seconds) - fast check
                    self.timers["health"] = QtCore.QTimer()
                    self.timers["health"].timeout.connect(self.refresh_health)
                    self.timers["health"].start(5000)

                    # Metrics timer (15 seconds)
                    self.timers["metrics"] = QtCore.QTimer()
                    self.timers["metrics"].timeout.connect(self.refresh_metrics)
                    self.timers["metrics"].start(15000)

                    # Logs timer (10 seconds) - reduced from 3s to avoid UI blocking
                    self.timers["logs"] = QtCore.QTimer()
                    self.timers["logs"].timeout.connect(self.refresh_logs)
                    self.timers["logs"].start(10000)

                    # Usage timer (15 seconds)
                    self.timers["usage"] = QtCore.QTimer()
                    self.timers["usage"].timeout.connect(self.refresh_usage)
                    self.timers["usage"].start(15000)

                # ===== CONTROL ACTIONS =====

                def test_connection(self):
                    """Test connection to cloud server and show result"""
                    import urllib.request

                    self.log("Testing connections...")

                    # Test cloud
                    cloud_ok = False
                    endpoints = [
                        ("http://localhost:8080/health", "localhost:8080"),
                        ("https://conjure.lautrek.com/health", "conjure.lautrek.com"),
                    ]
                    for url, endpoint in endpoints:
                        try:
                            start = time.time()
                            req = urllib.request.Request(url, method="GET")
                            req.add_header("User-Agent", "Conjure-FreeCAD/0.1.0")
                            with urllib.request.urlopen(req, timeout=5) as resp:
                                if resp.status == 200:
                                    latency = int((time.time() - start) * 1000)
                                    self.log(f"✓ Cloud server OK: {endpoint} ({latency}ms)")
                                    cloud_ok = True
                                    break
                        except Exception as e:
                            self.log(f"✗ {endpoint}: {e}")

                    if not cloud_ok:
                        self.log("✗ Could not connect to cloud server")

                    # Test bridge
                    bridge_health = self.get_real_health_status()
                    if bridge_health.get("status") == "success":
                        self.log("✓ FreeCAD bridge OK")
                    else:
                        self.log(f"✗ FreeCAD bridge: {bridge_health.get('message', 'not responding')}")

                    self.refresh_health()

                def reload_bridge(self):
                    """Restart the internal socket server with module reload"""
                    import importlib
                    import socket

                    self.log("Restarting socket server...")

                    # Stop existing socket server
                    if self.socket_server:
                        try:
                            self.socket_server.stop()
                            # Wait for the server thread to finish
                            if hasattr(self.socket_server, "thread") and self.socket_server.thread:
                                self.socket_server.thread.join(timeout=2.0)
                            self.log("✓ Socket server stopped")
                        except Exception as e:
                            self.log(f"WARNING: Error stopping server: {e}")
                        self.socket_server = None

                    # Wait for port to be released and verify it's free
                    port_free = False
                    for _attempt in range(10):  # Try for up to 2 seconds
                        time.sleep(0.2)
                        try:
                            test_sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
                            test_sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
                            test_sock.bind(("localhost", 9876))
                            test_sock.close()
                            port_free = True
                            break
                        except OSError:
                            pass  # Port still in use

                    if not port_free:
                        self.log("WARNING: Port 9876 still in use, attempting restart anyway...")

                    # Reload the conjure module to pick up code changes
                    try:
                        import conjure

                        # Also reload dependent modules in src/
                        modules_to_reload = ["rendering", "validation", "geometry", "history"]
                        for mod_name in modules_to_reload:
                            try:
                                mod = __import__(mod_name)
                                importlib.reload(mod)
                                self.log(f"✓ Reloaded {mod_name}")
                            except Exception:
                                pass  # Module may not exist or not be loaded

                        # Reload main conjure module
                        importlib.reload(conjure)
                        self.log("✓ Reloaded conjure module")

                        # Create and start new server
                        self.socket_server = conjure.ConjureServer()
                        self.socket_server.start()
                        self.log("✓ Socket server restarted on localhost:9876")

                    except Exception as e:
                        self.log(f"ERROR: Failed to restart server: {e}")
                        import traceback

                        self.log(traceback.format_exc())

                    time.sleep(0.3)
                    self.refresh_health()

                def log(self, message):
                    """Log message to console and Logs tab"""
                    print(f"INFO:Conjure_Gui:{message}")
                    # Also update logs tab if it exists
                    if hasattr(self, "logs_text"):
                        from datetime import datetime

                        timestamp = datetime.now().strftime("%H:%M:%S")
                        self.logs_text.append(f"[{timestamp}] {message}")

                # ===== HELPERS =====

                def escape_html(self, text):
                    """Escape HTML special characters"""
                    return text.replace("&", "&amp;").replace("<", "&lt;").replace(">", "&gt;")

                # ===== CLEANUP =====

                def getStandardButtons(self):
                    """Return standard buttons for the task panel"""
                    return int(QtWidgets.QDialogButtonBox.Close)

                def accept(self):
                    """Close the panel and cleanup"""
                    for timer in self.timers.values():
                        timer.stop()
                    # Stop socket server if it's running
                    if self.socket_server:
                        with contextlib.suppress(builtins.BaseException):
                            self.socket_server.stop()
                    Gui.Control.closeDialog()

                def reject(self):
                    """Close the panel and cleanup"""
                    for timer in self.timers.values():
                        timer.stop()
                    # Stop socket server if it's running
                    if self.socket_server:
                        with contextlib.suppress(builtins.BaseException):
                            self.socket_server.stop()
                    Gui.Control.closeDialog()

            # Create dashboard as a dockable widget instead of Task Panel
            mw = Gui.getMainWindow()
            if mw:
                # Check if dock already exists
                existing_dock = mw.findChild(QtWidgets.QDockWidget, "ConjureDashboard")
                if existing_dock:
                    existing_dock.show()
                    existing_dock.raise_()
                    print("INFO:Conjure_Gui:✓ Dashboard shown (existing)")
                else:
                    # Create new dock widget
                    mcp_panel = MCPDashboard()
                    dock = QtWidgets.QDockWidget("Conjure Dashboard", mw)
                    dock.setObjectName("ConjureDashboard")
                    dock.setWidget(mcp_panel.form)
                    dock.setAllowedAreas(QtCore.Qt.LeftDockWidgetArea | QtCore.Qt.RightDockWidgetArea)
                    dock.setFeatures(
                        QtWidgets.QDockWidget.DockWidgetClosable
                        | QtWidgets.QDockWidget.DockWidgetMovable
                        | QtWidgets.QDockWidget.DockWidgetFloatable
                    )
                    mw.addDockWidget(QtCore.Qt.RightDockWidgetArea, dock)
                    dock.show()
                    dock.raise_()
                    # Store reference to prevent garbage collection
                    mw._conjure_panel = mcp_panel
                    mw._conjure_dock = dock
                    print("INFO:Conjure_Gui:✓ Dashboard opened (dock widget)")
        except Exception as e:
            print(f"ERROR:Conjure_Gui:Failed to open dashboard: {e}")
            import traceback

            traceback.print_exc()


# Register the command
if not hasattr(Gui, "conjure_command"):
    Gui.conjure_command = ConjureShowCommand()
    Gui.addCommand("Conjure_Show", Gui.conjure_command)
    print("INFO:Conjure_Gui:✓ Registered Conjure command")


class ConjureWorkbench(Gui.Workbench):
    MenuText = "Conjure"
    ToolTip = "Conjure - AI CAD Control"

    def GetIcon(self):
        """Return the icon for this workbench"""
        user_dir = App.getUserAppDataDir()
        icon_path = os.path.join(user_dir, "Mod", "conjure", "assets", "conjure_icon.svg")
        return icon_path

    def Initialize(self):
        """This function is called at workbench creation."""
        print("INFO:Conjure_Gui:Initializing Conjure Workbench")

        # Set up command list
        self.command_list = ["Conjure_Show"]

        # Try to set up toolbar and menu, but don't fail if they don't work
        try:
            self.appendToolbar("Conjure Tools", self.command_list)
            print("INFO:Conjure_Gui:✓ Toolbar created")
        except:
            print("INFO:Conjure_Gui:Toolbar skipped (not available)")

        try:
            self.appendMenu("&Conjure", self.command_list)
            print("INFO:Conjure_Gui:✓ Menu created")
        except:
            print("INFO:Conjure_Gui:Menu skipped (not available)")

        print("INFO:Conjure_Gui:✓ Workbench initialized")

    def Activated(self):
        """This function is called when the workbench is activated."""
        print("INFO:Conjure_Gui:✓ Conjure Workbench activated")
        print("INFO:Conjure_Gui:  MCP Agent is running as a background service")
        print("INFO:Conjure_Gui:  Ready to receive commands from Claude Code")

    def Deactivated(self):
        """This function is called when the workbench is deactivated."""
        print("INFO:Conjure_Gui:Conjure Workbench deactivated")

    def GetClassName(self):
        """Return the name of the associated C++ class."""
        return "Gui::PythonWorkbench"


# Add the workbench if it hasn't been added already
if not hasattr(Gui, "conjure_workbench"):
    try:
        Gui.conjure_workbench = ConjureWorkbench()
        Gui.addWorkbench(Gui.conjure_workbench)
        print("INFO:Conjure_Gui:✓ Registered Conjure Workbench")
    except Exception as e:
        print(f"ERROR:Conjure_Gui:Failed to register Conjure Workbench: {e}")
